; -*- Mode: Scheme; Syntax: Scheme; Package: Scheme; -*-
; File record.scm / Copyright (c) 1989 Jonathan Rees / See file COPYING

;;;; Record package for Pseudoscheme

; Someday make this do (funcall (compile nil '(lambda () (defstruct ...)))).
; Or better, build this into pseudoscheme.

(lisp:defstruct (record-type-descriptor (:predicate #f)
					(:copier #f)
					(:conc-name #f)
					(:constructor make-rtd)
					(:print-function print-rtd))
  rtd-identification
  record-constructor-curried
  record-predicate
  record-accessor-curried
  record-updater-curried
  rtd-unique-id)

(if (lisp:fboundp 'scheme-internal:set-value-from-function)
    (scheme-internal:set-value-from-function (lisp:quote record-predicate)
				  'record-predicate))

(define (print-rtd rtd stream escape?)
  escape? ;ignored
  (lisp:format stream
	       "#{Record-type-descriptor ~S.~S}"
	       (rtd-identification rtd)
	       (rtd-unique-id rtd)))

(define record-type-table (lisp:make-hash-table :test 'lisp:equal))

(define (make-record-type type-id field-names)
  (let* ((key (cons type-id field-names))
	 (existing (lisp:gethash key record-type-table)))
    (if (and existing
	     (begin (lisp:format lisp:*query-io*
				 "~&Existing ~S has fields ~S.~%"
				 existing
				 field-names)
		    (lisp:y-or-n-p
		      "Use that descriptor (instead of creating a new one)? ")))
	existing
	(let ((new (really-make-record-type type-id field-names)))
	  (lisp:setf (lisp:gethash key record-type-table) new)
	  new))))

(define *record-type-unique-id* 0)

(define (really-make-record-type type-id field-names)

  (define size (+ (length field-names) 1))

  (define (constructor . init-names-option)
    (let* ((init-names (if (null? init-names-option)
			   field-names
			   (car init-names-option)))
	   (number-of-inits (length init-names))
	   (indexes (map field-index init-names)))
      (lambda field-values
	(if (= (length field-values) number-of-inits)
	    (let ((record (make-vector size)))
	      (vector-set! record 0 the-descriptor)
	      (for-each (lambda (index value)
			  (vector-set! record index value))
			indexes
			field-values)
	      record)
	    (lisp:error "Wrong number of arguments to ~S constructor -- ~S"
			type-id field-values)))))

  (define (predicate obj)
    (and (vector? obj)
	 (= (vector-length obj) size)
	 (eq? (vector-ref obj 0) the-descriptor)))

  (define (accessor name)
    (let ((i (field-index name)))
      (lambda (record)
	(if (eq? (vector-ref record 0) the-descriptor) ;(predicate record)
	    (vector-ref record i)
	    (lisp:error "Attempt to access the ~S of something ~
			that's not a ~S record -- ~S"
			name type-id record)))))

  (define (updater name)
    (let ((i (field-index name)))
      (lambda (record new-value)
	(if (eq? (vector-ref record 0) the-descriptor) ;(predicate record)
	    (vector-set! record i new-value)
	    (lisp:error "Attempt to assign the ~S of something ~
			that's not a ~S record -- ~S"
			name type-id record)))))

  (define (field-index name)
    (let loop ((l field-names) (i 1))
      (if (null? l)
	  (lisp:error "Not a field name for ~S records -- ~S" type-id name)
	  (if (eq? name (car l))
	      i
	      (loop (cdr l) (+ i 1))))))

  (define the-descriptor #f)

  (set! *record-type-unique-id* (+ *record-type-unique-id* 1))

  (set! the-descriptor
	(make-rtd :rtd-identification type-id
		  :record-constructor-curried constructor
		  :record-predicate predicate
		  :record-accessor-curried accessor
		  :record-updater-curried updater
		  :rtd-unique-id *record-type-unique-id*))

  the-descriptor)

(define (record-constructor r-t . rest)
  (apply (record-constructor-curried r-t) rest))

(define (record-accessor r-t field-name)
  ((record-accessor-curried r-t) field-name))

(define (record-updater r-t field-name)
  ((record-updater-curried r-t) field-name))
