/*----------------------------------------------------------------------

            T H E    P I N E    M A I L   S Y S T E M

   Laurence Lundblade and Mike Seibel
   Networks and Distributed Computing
   Computing and Communications
   University of Washington
   Administration Building, AG-44
   Seattle, Washington, 98195, USA
   Internet: lgl@CAC.Washington.EDU
             mikes@CAC.Washington.EDU

   Please address all bugs and comments to "pine-bugs@cac.washington.edu"

   Copyright 1989, 1990, 1991, 1992  University of Washington

    Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee to the University of
   Washington is hereby granted, provided that the above copyright notice
   appears in all copies and that both the above copyright notice and this
   permission notice appear in supporting documentation, and that the name
   of the University of Washington not be used in advertising or publicity
   pertaining to distribution of the software without specific, written
   prior permission.  This software is made available "as is", and
   THE UNIVERSITY OF WASHINGTON DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
   WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED
   WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN
   NO EVENT SHALL THE UNIVERSITY OF WASHINGTON BE LIABLE FOR ANY SPECIAL,
   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
   LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT
   (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION
   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
  

   Pine is in part based on The Elm Mail System:
    ***********************************************************************
    *  The Elm Mail System  -  $Revision: 2.13 $   $State: Exp $          *
    *                                                                     *
    * 			Copyright (c) 1986, 1987 Dave Taylor              *
    * 			Copyright (c) 1988, 1989 USENET Community Trust   *
    ***********************************************************************
 

  ----------------------------------------------------------------------*/



/*======================================================================
     newmail.c

   Check for new mail and queue up notification

 ====*/

#include "headers.h"


#ifdef ANSI
static void new_mail_mess(MAILSTREAM *, char *, long, long);
#else
static void new_mail_mess();
#endif


static int mailbox_mail_since_command = 0,
           inbox_mail_since_command   = 0;

/*----------------------------------------------------------------------
     new_mail() - check for new mail in the inbox
 
  Input:  cursor_moved -- address of flag that is set if the cursor was
                          moved due to checkpointing.
          force       -- flag indicating we should check for new mail no matter
          time_for_check_point -- 0: GoodTime, 1: BadTime, 2: VeryBadTime

  Result: returns -1 if there was no new mail. Otherwise returns the
          sorted message number of the smallest message number that
          was changed. That is the screen needs to be repainted from
          that message down.

  Limit frequency of checks because checks use some resources. That is
  they generate an IMAP packet or require locking the local mailbox.
  (Acutally the lock isn't required, a stat will do, but the current
   c-client mail code locks before it stats.)

  Returns true only if there is a change in the given mail stream. Otherwise
  this returns false. As of returning the messgae counts in the pine
  structure are updated to reflect the current number of messages including
  any new mail and any expunging.
 
 --- */
long
new_mail(cursor_moved, force, time_for_check_point)
     int *cursor_moved, force, time_for_check_point;
{
    static long last_check = 0;
    long        now, x, y, *old_sort, min_changed_sorted_msgno,
                old_current_raw_msgno, rv;
    MAILSTREAM *stream;
    register struct pine *pine_state;


    dprint(9, (debugfile, "new mail called (%x %d %d)\n",
               cursor_moved, force, time_for_check_point));

    pine_state = ps_global;  /*  this gets called out of the composer which
                                 doesn't know about pine_state
                              */
    stream = pine_state->mail_stream;

    if(stream == NULL)
      return(-1);

    now = time(0);

    if(!force && now-last_check < timeout-2 && !pine_state->mail_box_changed) 
      return(-1);            /* -2 so we're sure to check every timeout */

    dprint(6, (debugfile, "***** new_mail pinging *****\n"));
    

    /*-- Ping the mailstream to check for new mail --*/
    dprint(6, (debugfile, "New mail checked \n"));
    last_check = now;

/*    StartInverse(); 
    PutLine0(0,1,"++"); *//* Show something on the screen to indicate delay*/
/*    MoveCursor(ps_global->ttyo->screen_rows -3, 0);
    fflush(stdout); */

    if((char *)mail_ping(stream) == NULL)
      pine_state->dead_stream = 1;

/*    if(cursor_moved != NULL)
      *cursor_moved = 1; 
    PutLine0(0, 1, "  ");
    EndInverse();
    fflush(stdout); */

    if(stream != pine_state->inbox_stream && pine_state->inbox_stream != NULL)
      if((char *)mail_ping(pine_state->inbox_stream) == NULL)
        pine_state->dead_inbox = 1;


    /*-------------------------------------------------------------
       Mail box state changed, could be additions or deletions.
      -------------------------------------------------------------*/
    if(pine_state->mail_box_changed) {
        dprint(7, (debugfile,
        "New mail, %s,  new_mail_count:%d  expunge count:%d,  max_msgno:%d\n",
                   pine_state->mail_stream == pine_state->inbox_stream ?
                      "inbox" : "other",
                   pine_state->new_mail_count,
                   pine_state->expunge_count,
                   pine_state->max_msgno));

        if(ps_global->new_mail_count > 0) {
            mailbox_mail_since_command += ps_global->new_mail_count;

            new_mail_mess(pine_state->mail_stream,
                      pine_state->mail_stream == pine_state->inbox_stream ?
                      NULL :  pine_state->cur_folder,
                      mailbox_mail_since_command,
                      pine_state->max_msgno);
        }

        old_current_raw_msgno =
                         ps_global->sort[ps_global->current_sorted_msgno];
        dprint(9, (debugfile,
                  "old_current_raw_msgno: %ld  sort_allocated; %ld\n",
                   old_current_raw_msgno, ps_global->sort_allocated));

        if(ps_global->sort_allocated <= ps_global->max_msgno + 1){
            /*--- Pad out the sort array if needed ----*/
            ps_global->sort_allocated = max(ps_global->max_msgno + 1,
                                            ps_global->sort_allocated * 2);
            fs_resize((void **)&(ps_global->sort),
                      ps_global->sort_allocated * sizeof(long));
        }

        for(x  = ps_global->max_msgno - ps_global->new_mail_count + 1;
            x <= ps_global->max_msgno; x++) {
            ps_global->sort[x] = x;
        }

        if(ps_global->sort_allocated) {
           old_sort = (long *)fs_get(sizeof(long)*ps_global->sort_allocated);
           for(x = 1; x <= ps_global->max_msgno; x++)
             old_sort[x] = ps_global->sort[x];
        }  else {
            old_sort = NULL;
        }


        sort_current_folder();

        /* Figure out what the current message number is now */
        for(x = 1; x <= ps_global->max_msgno; x++) {
            if(ps_global->sort[x] == old_current_raw_msgno) {
                ps_global->new_current_sorted_msgno = x;
                break;
            }
        }
        /* Figure out the lowest position message sorted too */
        if(old_sort != NULL)
          for(x = 1; x <= ps_global->max_msgno; x++) {
              if(ps_global->sort[x] != old_sort[x])
                break;
          }     
        /* Figure out the lowest position that changed */
        if(pine_state->expunge_count > 0 || old_sort == NULL) {
            min_changed_sorted_msgno = 0;
        } else if(x > ps_global->max_msgno) {
            min_changed_sorted_msgno = ps_global->max_msgno -
              ps_global->new_mail_count;
        } else {
            min_changed_sorted_msgno = x - 1;
        }
        dprint(9, (debugfile, "new sorted_msgno: %ld  min_changed: %ld\n",
                   ps_global->new_current_sorted_msgno,
                   min_changed_sorted_msgno));

        if(old_sort != NULL)
          fs_give((void **)&old_sort);

        clear_index_cache();
    }

    if(pine_state->inbox_changed &&
                       pine_state->inbox_stream != pine_state->mail_stream) {
        /*--  New mail for the inbox, queue up the notification           --*/
        /*-- If this happens then inbox is not current stream that's open --*/
        dprint(7, (debugfile,
         "New mail, inbox, new_mail_count:%ld expunge: %ld,  max_msgno %ld\n",
                   pine_state->inbox_new_mail_count,
                   pine_state->inbox_expunge_count,
                   pine_state->inbox_max_msgno));

        if(pine_state->inbox_new_mail_count > 0) {
            inbox_mail_since_command       += ps_global->inbox_new_mail_count;
            ps_global->inbox_new_mail_count = 0;
            new_mail_mess(pine_state->inbox_stream,NULL,
                          inbox_mail_since_command,
                          pine_state->inbox_max_msgno);
        }
    }


    rv = pine_state->mail_box_changed ? min_changed_sorted_msgno : -1;
    pine_state->inbox_changed    = 0;
    pine_state->mail_box_changed = 0;
    ps_global->new_mail_count    = 0;


    if(check_point(0, time_for_check_point == 0 ? GoodTime:
                      time_for_check_point == 1 ? BadTime : VeryBadTime)) {
        if(cursor_moved != NULL) {
            dprint(9, (debugfile, "newmail() set cursor moved flag\n"));
            *cursor_moved = 1;
        }
    } else {
        if(cursor_moved != NULL) {
            dprint(9, (debugfile, "newmail() cleared cursor moved flag\n"));
            *cursor_moved = 0;
        }
    }

    dprint(6, (debugfile, "******** new mail returning %ld  ********\n", rv));
    return(rv);
}


/*----------------------------------------------------------------------
     Format and queue a "new mail" message

  Args: stream     -- mailstream on which a mail has arrived
        folder     -- Name of folder, NULL if inbox
        number     -- number of new messages since last command
        max_num    -- The number of messages now on stream

 Not too much worry here about the length of the message because the
status_message code will fit what it can on the screen and truncation on
the right is about what we want which is what will happen.
  ----*/
static void
new_mail_mess(stream, folder, number, max_num)
     MAILSTREAM *stream;
     long        number, max_num;
     char       *folder;

{
    ENVELOPE *e;
    char      subject[200], from[MAX_ADDRESS+20], intro[50 + MAXFOLDER];
    static char   *carray[] = { "regarding",
				"concerning",
				"about",
				"as to",
				"as regards",
				"as respects",
				"in re",
				"re",
				"respecting",
				"in point of",
				"with regard to",
				"subject:"
    };

    e = mail_fetchenvelope(stream, (long)max_num);

    if(folder == NULL) {
        if(number > 1)
          sprintf(intro, "%ld new messages!", number);
        else
          strcpy(intro, "New mail!");
    } else {
        if(number > 1)
          sprintf(intro,"%ld messages saved to folder \"%s\"", number, folder);
        else
          sprintf(intro, "Mail saved to folder \"%s\"", folder);
    }
     
    if(number > 1) {
        if(e != NULL && e->from != NULL) {
            if(e->from->personal != NULL)
              sprintf(from, " Most recent from %s", e->from->personal);
            else
              sprintf(from, " Most recent from %s@%s", e->from->mailbox,
                      e->from->host);
        } else {
            from[0] = '\0';
        }
        q_status_message2(0, 2, 4, "\007%s%s", intro, from);
    } else {
        if(e != NULL && e->from != NULL) {
            if(e->from->personal != NULL)
              sprintf(from, " From %s", e->from->personal);
            else
              sprintf(from, " From %s@%s", e->from->mailbox, e->from->host);
        } else {
            from[0] = '\0';
        }
        if(e != NULL && e->subject == NULL)
          strcpy(subject, " with no subject");
        else
          sprintf(subject, " %s %.199s", carray[random() %12 ], e->subject);
        if(from == NULL)
          subject[1] = toupper(subject[1]);

        q_status_message3(0, 2,4, "\007%s%s%s", intro, from, subject);
    }
}



/*----------------------------------------------------------------------
    Force write of the main file so user doesn't loose too much when
 something bad happens. The only thing that can get lost is the 'D' status,
 as new mail written on disk when delivered.

 Args: change made -- indicates that change count should go up
       timing      -- indicates if it's a good time for to do a checkpoine

  Result: returns 1 if checkpoint was written, 
                  0 if not.

Only need to checkpoint current stream because there are no changes going
on with other streams because we're not manipulating them.
  ----*/
static int check_count = 0;
static long last_check_point = 0;

check_point(change_made, timing)
     int            change_made;
     CheckPointTime timing;
{
    int freq, tm;

    dprint(9, (debugfile, "check_point(%d, %s)\n", change_made,
               timing == GoodTime ? "GoodTime" :
               timing == BadTime  ? "BadTime" : "VeryBadTime"));

    freq = CHECK_POINT_FREQ * (timing==GoodTime ? 1 : timing==BadTime ? 3 : 4);
    tm   = CHECK_POINT_TIME * (timing==GoodTime ? 1 : timing==BadTime ? 2 : 4);

    dprint(9, (debugfile, "freq: %d  count: %d\n", freq, check_count));
    dprint(9, (debugfile, "tm: %d  elapsed: %d\n", tm,
               time(0) - last_check_point));

    if(change_made)
      check_count++;

    if(!check_count)
      last_check_point = time(0);

    if(ps_global->mail_stream != NULL && !ps_global->mail_stream->readonly &&
       (check_count >= freq || 
       (check_count > 0 && time(0) - last_check_point >= tm))){
        dprint(2, (debugfile, "Doing checkpoint\n"));
        StartInverse();
        PutLine0(0,1,"**"); /* Show something on the screen to indicate delay*/
	MoveCursor(ps_global->ttyo->screen_rows -3, 0);
        fflush(stdout);
        mail_check(ps_global->mail_stream);
                                    /* Causes mail file to be written */
        
        check_count = 0;
        last_check_point = time(0);
        PutLine0(0, 1, "  ");
        EndInverse();
        fflush(stdout);
        return(1);
    } else {
        return(0);
    }
}




/*----------------------------------------------------------------------
    Call this when a mail file is written to reset timer and counter
  for next check_point.
  ----------------------------------------------------------------------*/
void
reset_check_point()
{
    check_count = 0;
    last_check_point = time(0);
}



/*----------------------------------------------------------------------
    Zero the counters that keep track of mail accumulated between
   commands.
 ----*/
void
zero_new_mail_count()
{
    dprint(9, (debugfile, "New_mail_count zeroed\n"));
    mailbox_mail_since_command = 0;
    inbox_mail_since_command   = 0;
}


/*----------------------------------------------------------------------
     Check and see if all the stream are alive

Returns:  0 if there was no change
          1 if streams have died since last call

Also outputs a message that the streams have died
 ----*/
streams_died()
{
    int rv = 0, inbox = 0;

    if(ps_global->dead_stream && !ps_global->noticed_dead_stream) {
        rv = 1;
        ps_global->noticed_dead_stream = 1;
        if(ps_global->mail_stream == ps_global->inbox_stream)
          ps_global->noticed_dead_inbox = 1;
    }

    if(ps_global->dead_inbox && !ps_global->noticed_dead_inbox) {
        rv = 1;
        ps_global->noticed_dead_inbox = 1;
        inbox = 1;
    }
    if(rv == 1) 
      q_status_message1(1, 3, 6,
                        "\007MAIL FOLDER \"%s\" CLOSED DUE TO ACCESS ERROR",
                        pretty_fn(inbox? ps_global->inbox_name :
                                         ps_global->cur_folder));
    return(rv);
}
        
