/*
 * rdm_column.c - routines to read multi-column curve data from an MTV-format.
 *
 * The data is assumed to be sandwiched between $DATA=xxx lines
 */

#include <stdio.h>
#include <math.h>
#include <strings.h>
#include <malloc.h>
#include "CNplot.h"
#include <stdlib.h> /*<mkl>*/
/*
 * Data is organized in columns:
 *     Column_1   Column_2 ... Column_n
 *
 * Each column contains a number of row-sets
 *     Column_1
 *       row-set-1 
 *       row-set-2 
 *       ...           
 *       row-set-m
 *
 * Each row-set contains several rows of double-precision values
 *     Column_1
 *       row-set-1 {row1 row2 ... row-m}
 *       row-set-2 {row1 row2 ... row-j}
 *       ...           
 *       row-set-m
 *
 */    

/* Row Data-structure */
typedef struct CNrow_strct {
   double value;
   struct CNrow_strct *next;
   struct CNrow_strct *prev;
} CNrow;
typedef struct CNrow_strct *CNrowptr;

/* Row-Set Data-structure */
typedef struct CNrowset_strct {
   CNcurve_property *curv_pr;
   struct CNrow_strct    *rowhead;
   struct CNrow_strct    *rowtail;
   struct CNrowset_strct *next;
   struct CNrowset_strct *prev;
} CNrowset;
typedef struct CNrowset_strct *CNrowsetptr;
    
/* Column Data-structure */
typedef struct CNcolumn_strct {
   char   name[CN_MAXCHAR];
   struct CNrowset_strct *rowsethead;
   struct CNrowset_strct *rowsettail;
   struct CNcolumn_strct *next;
   struct CNcolumn_strct *prev;
} CNcolumn;
typedef struct CNcolumn_strct *CNcolumnptr;

int CNmtv_read_column_data();
static void        read_cl_options();
static void        column_itemno_err();

static CNcolumnptr make_column();
static CNcolumnptr CNinsert_column();
static void        CNdelete_column();
static void        CNdelete_column_list();

static CNrowsetptr make_rowset();
static CNrowsetptr CNinsert_rowset();
static void        CNdelete_rowset();
static void        CNdelete_rowset_list();
static void        CNapply_curvpr_to_rowset();

static CNrowptr    make_row();
static CNrowptr    CNinsert_row();
static void        CNdelete_row();
static void        CNdelete_row_list();

/* 
 * Read data in multicolumn format
 * Return an EOF if such is found during the read; otherwise return 0
 */
/*ARGSUSED*/
int CNmtv_read_column_data(datahead, datatail, dataID, dataname,
                           filename, fp, lineno, header, verbose)
CNdslistptr  *datahead;        /* Pointers to plot data                */
CNdslistptr  *datatail;        /* Pointers to plot data                */
int          *dataID;          /* Data ID                              */
char         *dataname;        /* Dataset name                         */
char         *filename;        /* The name of the file/pipe source     */
FILE         *fp;              /* The file/pipe source                 */
int          *lineno;          /* Current line number                  */
char         *header;          /* Header string                        */
int          verbose;          /* Verbosity Flag                       */
{
   char               *CNstrip_keyword();
   CNcontstepptr      cstephead=NULL, csteptail=NULL;
   CNannotptr         annothead=NULL, annottail=NULL;
   CNcolumnptr        column_listhead=NULL, column_listtail=NULL, C, CX;
   CNrowsetptr        RS, RSx, RSy;
   CNrowptr           Rx, Ry;
   CNcurveptr         curvehead, curvetail, Cptr;
   CNpointptr         P;
   CNdatasetptr       dptr;
   CNviewptr          view_pr;
   CNcurve_property   cv_property;
   CNgbcurve_property gb_property;
   CNdataset_property ds_property;
   CNplotset_property pt_property;
   char               line[BUFSIZ];
   char               xcolumn[CN_MAXCHAR];
   char               name[CN_MAXCHAR];
   char               *word[CN_MAXWORDS];
   int                header_found = CN_FALSE;
   int                nw=0, len, ierr=0, i, FOUND=CN_FALSE;
   int                first_dataset = CN_TRUE;
   int                pointID, curveID;
   double             value, x, y, z;
   double             xmin, xmax, ymin, ymax, zmin=0.0, zmax=0.0;
 
   /* Print info */
   (void) fprintf(stdout,"\n   Reading COLUMN data...(line %d)\n",*lineno);

   /* Set the view parameters */
   view_pr = CNcreate_view();

   /* Set the properties of the plotset */
   CNset_default_plotset_property(&pt_property);

   /* Set the properties of the dataset */
   CNset_default_dataset_property(&ds_property);

   /* Set the properties of the curve */
   CNset_default_curve_property(&cv_property);

   /* Set the properties of the global curve */
   CNset_default_gbcurve_property(&gb_property);

   /* Initalize */
   (void) strcpy(xcolumn,"");

   /* Keep on reading until a "$" is encountered */
   while (!header_found && CNgetucline(fp, line, lineno, BUFSIZ) != EOF) {

      if (((len=strlen(line)) == 0) || (len==1 && line[0]=='\n')) {
         /* Create new row-sets */
         for (C=column_listhead; C!=NULL; C=C->next) {
            if (C->rowsettail != NULL && C->rowsettail->rowtail != NULL) {
               /* Apply the curve properties to the rowset */
               CNapply_curvpr_to_rowset(C->rowsettail, &cv_property);

               /* Create a new rowset */
               (void) CNinsert_rowset(&(C->rowsethead), &(C->rowsettail));
            }
         }

         /* New curve */
         CNdelete_curve_property_fields(&cv_property);
         CNset_default_curve_property(&cv_property);
 
      /*EMPTY*/
      } else if (line[0] == '#') {
         /* Comment */
         ;
 
      } else if (line[0] == '$') {
         /* Header */
         line[0] = ' ';
         (void) strcpy(header, line);
         header_found = CN_TRUE;
 
      } else if (line[0] == '@') {
         /* Annotation */
         line[0] = ' ';
         CNparse_annotation_line(line, BUFSIZ,
                                 &annothead, &annottail,
                                 verbose);
 
      } else if (line[0] == '%') {
         /* Option  */
         line[0] = ' ';
         read_cl_options(line,view_pr,&pt_property,&ds_property,
                         &cv_property, &gb_property, xcolumn, verbose);
         if ((ds_property.contours) && ((ds_property.flag & CNctrlevel)!=0))
         CNset_contstep_levels(&cstephead, &csteptail, &ds_property);

      } else {
         /* Read column data */
         
         if ((nw = CNgetwords(line,word, CN_MAXWORDS)) >= 2) {
            if (column_listhead == NULL) {
               /* make columns */
               for (i=0; i<nw; i++)
                  (void) CNinsert_column(&column_listhead, &column_listtail, 
                                         CNstrip_keyword(word[i]));
            } else {
               /* Add to the columns */
               i = 0;
               for (C=column_listhead; C!=NULL; C=C->next) {
                  value = (i < nw) ? atof(word[i]) : 0.0; 
                  if ((RS=C->rowsettail) == NULL) {
                     /* Create a rowset */
                     RS = CNinsert_rowset(&(C->rowsethead), &(C->rowsettail));
                  }
                  if (RS != NULL) {
                     /* Put the row into this rowset */
                     (void) CNinsert_row(&(RS->rowhead), &(RS->rowtail),value);
                  }
                  i++;
               }      
            }
         } else {
            column_itemno_err(*lineno,2,ierr++);
         }

         /* Free the words */
         CNfreewords(&nw,word);
      }
   }

   /* Create the various datasets */
   if (column_listhead != NULL) {

      /* Find the x-column */
      CX = column_listhead;
      if (strlen(xcolumn) > 0) {
         for (C=column_listhead; C!=NULL && !FOUND; C=C->next) {
            if (strcmp(C->name,xcolumn)==0) {
               FOUND = CN_TRUE;
               CX = C;
            }
         }
      }

      /* Separate out the curves */
      for (C=column_listhead; C!=NULL; C=C->next) {
         if (C==CX) continue;
         if (C->rowsethead==NULL) continue;

         curveID   = 0;
         pointID   = 0;
         curvehead = NULL;
         curvetail = NULL;
         (void) sprintf(name,"%s vs %s",C->name,CX->name);
         RSy = C->rowsethead;
         for (RSx=CX->rowsethead; RSx!=NULL; RSx=RSx->next) {

            /* Don't create the curve if the rowset is empty */
            if (RSx->rowhead == NULL) continue;

            /* Create a curve */
            Cptr = CNinsert_curve(&curvehead, &curvetail, curveID++);

            /* Set curve properties */
            if (RSx->curv_pr)
            CNset_curve_property(&(Cptr->curv_pr), RSx->curv_pr);

            /* If the label on the curve has not been set, set it */
            if ((Cptr->curv_pr.flag & CNlinelabel) == 0) {
               (void) CNparse_curve_property(&(Cptr->curv_pr),
                                             "linelabel", C->name, 0);
            }

            /* Fill the curve */
            Ry = RSy->rowhead;
            for (Rx=RSx->rowhead; Rx!=NULL; Rx=Rx->next) {
               x = Rx->value; 
               y = (Ry!=NULL) ? Ry->value : 0.0;
               z = 0.0;
               (void) CNinsert_point(&(Cptr->pointhead), &(Cptr->pointtail), 
                                     x, y, z, pointID++);
               if (Ry != NULL) Ry = Ry->next;
            }
            if (RSy != NULL) RSy = RSy->next;
         } 

         /* Now create the dataset */
         if (curvehead != NULL) {

            /* 
             * Apply the global-curve dataset properties to 
             * the individual curves 
             */
            if (gb_property.flag != 0)
            CNreset_curves(curvehead, curvetail, &gb_property, 0);

            /* Get the min and max of the curves */
            xmin =  CN_LARGE;
            xmax = -CN_LARGE;
            ymin =  CN_LARGE;
            ymax = -CN_LARGE;
            for (Cptr=curvehead; Cptr!=NULL; Cptr=Cptr->next) {
               for (P=Cptr->pointhead; P!=NULL; P=P->next) {
                  if (P->x < xmin) xmin = P->x;
                  if (P->x > xmax) xmax = P->x;
                  if (P->y < ymin) ymin = P->y;
                  if (P->y > ymax) ymax = P->y;
               }
            }

            /* Adjust xmin, xmax, ymin, ymax, zmin, zmax */
            CNmtv_adjust_boundaries(&xmin, &xmax, &ymin, &ymax, &zmin, &zmax);

            /*
             * If the plot-boundary has been set in the options then
             * use that as the true plot boundary
             */
            if ((pt_property.flag & CNvxmin) != 0) xmin = pt_property.vxmin;
            if ((pt_property.flag & CNvymin) != 0) ymin = pt_property.vymin;
            if ((pt_property.flag & CNvzmin) != 0) zmin = pt_property.vzmin;
            if ((pt_property.flag & CNvxmax) != 0) xmax = pt_property.vxmax;
            if ((pt_property.flag & CNvymax) != 0) ymax = pt_property.vymax;
            if ((pt_property.flag & CNvzmax) != 0) zmax = pt_property.vzmax;

            /* Create a dataset containing the curves */
            dptr = CNmake_dataset(filename,name,CN_PLOT2D,
                                  xmin,xmax,ymin,ymax,zmin,zmax,
                                  xmin,xmax,ymin,ymax,zmin,zmax,*dataID);
            if (dptr != NULL) (*dataID)++;

            if (dptr != NULL) {
               dptr->curvehead = curvehead;
               dptr->curvetail = curvetail;

               /* Set overlay */
               /*
               (void) CNparse_plotset_property(&(dptr->plot_pr),
                                            "overlay", "FALSE", 0);
                */

               /* Set name */
               (void) CNparse_plotset_property(&(dptr->plot_pr),
                                            "toplabel", name, 0);

               /* Apply the view options to the dataset */
               CNset_view_property(dptr->view_pr,view_pr);
 
               /* Apply the plotset options to the dataset */
               CNset_plotset_property(&(dptr->plot_pr),&pt_property);
 
               /* Apply the dataset options to the dataset */
               CNset_dataset_property(&(dptr->data_pr),&ds_property);

               /* Apply the contsteps */
               if (cstephead != NULL) {
                  if (first_dataset) {
                     dptr->cstephead = cstephead;
                     dptr->csteptail = csteptail;
                     first_dataset = CN_FALSE;
                  } else {
                     CNcopy_contstep_list(&(dptr->cstephead),
                                          &(dptr->csteptail),
                                          cstephead, csteptail);
                  }
               }

               /* Print out the dataset */
               if (verbose) CNprint_dataset(dptr, 0);
 
#ifdef DEBUG
               /* Print out the curves */
               CNprint_curve_list(dptr->curvehead, dptr->curvetail, 1);
#endif

               /* Store the dataset in the dataset linked list */
               (void) CNinsert_dslist(datahead, datatail, dptr);
            }
         }
      }
   }
         
   /* Delete all the columns */
   CNdelete_column_list(&column_listhead, &column_listtail);

   /* Reset the property structures */
   CNdelete_plotset_property_fields(&pt_property);
   CNdelete_dataset_property_fields(&ds_property);
   CNdelete_gbcurve_property_fields(&gb_property);
   CNdelete_curve_property_fields(&cv_property);

   /* Free the view structure */
   CNdelete_view(view_pr);

   /* return */
   if (!header_found)
      return(EOF);
   else
      return(header_found);
}



/*
 * READING UTILITIES FOR 2D/3D CURVES
 */

/* 
 * Read a line and apply options to a dataset or a curve
 */
static void read_cl_options(header,
                            view_pr,pt_prop,ds_prop,cv_prop,gb_prop,
                            xcol,vbs)
char               *header;
CNviewptr          view_pr;
CNplotset_property *pt_prop;
CNdataset_property *ds_prop;
CNcurve_property   *cv_prop;
CNgbcurve_property *gb_prop;
char               *xcol;
int                vbs;
{
   char *argtbl[CN_MAXWORDS], *valtbl[CN_MAXWORDS];
   char newheader[CN_MAXCHAR];
   int  nargs = 0, nvals = 0;
   int  argfound, i;

   /* CNparse_line wants "command arg=val arg=val" so create a new header */
   (void) sprintf(newheader, "datafile %s",header);

   /* Get the argument-value pairs from the line */
   if (CNparse_line(newheader, CN_MAXCHAR,
                    &nargs, argtbl, CN_MAXWORDS,
                    &nvals, valtbl, CN_MAXWORDS)) {

      /* Go thru the arguments and find specialized matches */
      i = 0;
      while (i < nargs) {
 
         /* Go thru the arguments and find a match */
         argfound = CN_TRUE;
 
         if (strncmp(argtbl[i],"xcol",4)==0) {
            CNassign_string_keyword(xcol,valtbl[i],"xcolumn",0);
            if (vbs)
            (void)fprintf(stdout,"   Column  : %-14s= %s\n","xcol",xcol);

         } else {
            argfound = CN_FALSE;
         }
 
         /* Reset the table if a match was found; otherwise increment index */
         CNdownshift(argfound,&i,argtbl,valtbl,&nargs,&nvals);
      }

      /* Look for plotset/dataset/curve arguments */
      for (i=0; i<nargs; i++) {
         if (!CNparse_view_property       (view_pr,argtbl[i],valtbl[i],vbs))
           if (!CNparse_plotset_property  (pt_prop,argtbl[i],valtbl[i],vbs))
             if (!CNparse_dataset_property(ds_prop,argtbl[i],valtbl[i],vbs))
               if (!CNparse_gbcurve_property(gb_prop,argtbl[i],valtbl[i],vbs))
                 if (!CNparse_curve_property(cv_prop,argtbl[i],valtbl[i],vbs))
                    (void) fprintf(stderr,
                                "   warning : Invalid option \"%s=%s\"\n",
                                argtbl[i],valtbl[i]);
      }

      /* Clear the tables */
      CNfreewords(&nargs, argtbl);
      CNfreewords(&nvals, valtbl);
   }
}

/*
 * Print item-number error
 */
/*ARGSUSED*/
static void column_itemno_err(lineno,itemno,ierr)
int  lineno,itemno,ierr;
{
   (void) fprintf(stderr,"   warning (line %3d) %8s: ",lineno,"COLUMN");
   (void) fprintf(stderr,"need at least %d items\n",itemno);
}


/*
 * COLUMN DATA_STRUCT ALLOCATION
 */
 
/*
 * make a column
 */
static CNcolumnptr make_column(name)
char *name;
{
   CNcolumnptr newptr;
   unsigned int size = sizeof(CNcolumn);
 
   if ((newptr = (CNcolumnptr)malloc(size))!=NULL) {
      if (name != NULL) (void) strcpy(newptr->name,name);
      newptr->rowsethead = NULL;
      newptr->rowsettail = NULL;
      newptr->next       = NULL;
      newptr->prev       = NULL;
   }
 
   return(newptr);
}
 
/*
 * Insert a column at the tail of the current column list
 */
static
CNcolumnptr CNinsert_column(column_listhead,column_listtail,name)
CNcolumnptr *column_listhead, *column_listtail;
char     *name;
{
   static CNcolumnptr make_column();
   CNcolumnptr next,A,B;
 
   A = *column_listtail;
   if ((B=make_column(name))!=NULL) {
      if (A==NULL) {
         *column_listhead = B;
         *column_listtail = B;
      } else {
         next = A->next;
         B->next = next;
         B->prev = A;
         A->next = B;
         if (next    != NULL) next->prev = B;
         if (B->next == NULL) *column_listtail = B;
      }
   }
   return(B);
}
 
/*
 * Delete column
 */
static void CNdelete_column(column_listhead, column_listtail, C)
CNcolumnptr *column_listhead, *column_listtail;
CNcolumnptr C;
{
   CNcolumnptr prev,next;
 
   /* Delete the rowset */
   CNdelete_rowset_list(&(C->rowsethead), &(C->rowsettail));

   prev = C->prev;
   next = C->next;
   if (prev!=NULL) prev->next = next;
   if (next!=NULL) next->prev = prev;
   if (C==*column_listhead) *column_listhead = next;
   if (C==*column_listtail) *column_listtail = prev;
 
   /* Now delete C */
   free ((char*)C);
}

 
/*
 * Delete all the columns in the list
 */
static void CNdelete_column_list(column_listhead, column_listtail)
CNcolumnptr *column_listhead, *column_listtail;
{
   CNcolumnptr C;
 
   while ((C = *column_listhead) != NULL)
      CNdelete_column(column_listhead, column_listtail, C);
}


/*
 * ROW-SET DATA_STRUCT ALLOCATION
 */
 
/*
 * make a rowset
 */
static CNrowsetptr make_rowset()
{
   CNrowsetptr newptr;
   unsigned int size = sizeof(CNrowset);
 
   if ((newptr = (CNrowsetptr)malloc(size))!=NULL) {
      newptr->curv_pr   = NULL;
      newptr->rowhead   = NULL;
      newptr->rowtail   = NULL;
      newptr->next      = NULL;
      newptr->prev      = NULL;
   }
 
   return(newptr);
}
 
/*
 * Insert a rowset at the tail of the current rowset list
 */
static
CNrowsetptr CNinsert_rowset(rowset_listhead,rowset_listtail)
CNrowsetptr *rowset_listhead, *rowset_listtail;
{
   static CNrowsetptr make_rowset();
   CNrowsetptr next,A,B;
 
   A = *rowset_listtail;
   if ((B=make_rowset())!=NULL) {
      if (A==NULL) {
         *rowset_listhead = B;
         *rowset_listtail = B;
      } else {
         next = A->next;
         B->next = next;
         B->prev = A;
         A->next = B;
         if (next    != NULL) next->prev = B;
         if (B->next == NULL) *rowset_listtail = B;
      }
   }
   return(B);
}
 
/*
 * Delete rowset
 */
static void CNdelete_rowset(rowset_listhead, rowset_listtail, R)
CNrowsetptr *rowset_listhead, *rowset_listtail;
CNrowsetptr R;
{
   CNrowsetptr prev,next;
 
   /* Free the curve property if that exists */
   if (R->curv_pr != NULL) CNdelete_curve_property(R->curv_pr);

   /* Free the row list */
   CNdelete_row_list(&(R->rowhead), &(R->rowtail));

   prev = R->prev;
   next = R->next;
   if (prev!=NULL) prev->next = next;
   if (next!=NULL) next->prev = prev;
   if (R==*rowset_listhead) *rowset_listhead = next;
   if (R==*rowset_listtail) *rowset_listtail = prev;
 
   /* Now delete R */
   free ((char*)R);
}

 
/*
 * Delete all the rowsets in the list
 */
static void CNdelete_rowset_list(rowset_listhead, rowset_listtail)
CNrowsetptr *rowset_listhead, *rowset_listtail;
{
   CNrowsetptr R;
 
   while ((R = *rowset_listhead) != NULL)
      CNdelete_rowset(rowset_listhead, rowset_listtail, R);
}


/* 
 * Apply the curve properties to the rowset 
 */
static void CNapply_curvpr_to_rowset(R, curv_pr)
CNrowsetptr R;
CNcurve_property *curv_pr;
{

   /* Error check */
   if (R==NULL || curv_pr==NULL || curv_pr->flag==0) return;

   /* Create a curve-property structure */
   if (R->curv_pr == NULL) {
      R->curv_pr = CNmake_curve_property();
   }

   /* Apply the property */
   if (R->curv_pr != NULL) {
      CNset_curve_property(R->curv_pr, curv_pr);
   } 
}



/*
 * ROW DATA_STRUCT ALLOCATION
 */
 
/*
 * make a row
 */
static CNrowptr make_row(value)
double value;
{
   CNrowptr newptr;
   unsigned int size = sizeof(CNrow);
 
   if ((newptr = (CNrowptr)malloc(size))!=NULL) {
      newptr->value     = value;
      newptr->next      = NULL;
      newptr->prev      = NULL;
   }
 
   return(newptr);
}
 
/*
 * Insert a row at the tail of the current row list
 */
static
CNrowptr CNinsert_row(row_listhead,row_listtail,value)
CNrowptr *row_listhead, *row_listtail;
double   value;
{
   static CNrowptr make_row();
   CNrowptr next,A,B;
 
   A = *row_listtail;
   if ((B=make_row(value))!=NULL) {
      if (A==NULL) {
         *row_listhead = B;
         *row_listtail = B;
      } else {
         next = A->next;
         B->next = next;
         B->prev = A;
         A->next = B;
         if (next    != NULL) next->prev = B;
         if (B->next == NULL) *row_listtail = B;
      }
   }
   return(B);
}
 
/*
 * Delete row
 */
static void CNdelete_row(row_listhead, row_listtail, R)
CNrowptr *row_listhead, *row_listtail;
CNrowptr R;
{
   CNrowptr prev,next;
 
   prev = R->prev;
   next = R->next;
   if (prev!=NULL) prev->next = next;
   if (next!=NULL) next->prev = prev;
   if (R==*row_listhead) *row_listhead = next;
   if (R==*row_listtail) *row_listtail = prev;
 
   /* Now delete R */
   free ((char*)R);
}

 
/*
 * Delete all the rows in the list
 */
static void CNdelete_row_list(row_listhead, row_listtail)
CNrowptr *row_listhead, *row_listtail;
{
   CNrowptr R;
 
   while ((R = *row_listhead) != NULL)
      CNdelete_row(row_listhead, row_listtail, R);
}

