
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/**********************************************************************
*
*  File: sqcurve.c
*
*  Purpose: Does calculations needed for including square curvature
*           in energy. Linear model only.
*/

#include "include.h"

/* auxiliary structures needed to hold accumulated data */
/* per vertex structure */
struct v_curve { REAL area;    /* area around vertex */
                 REAL force[MAXCOORD];  /* force on vertex */
                 REAL deriv2[MAXCOORD][MAXCOORD];  /* self second deriv */
               } *v_curve;

/* per edge structure */
struct e_curve { REAL deriv[2][MAXCOORD]; /* dA_head/dv_tail */
                 REAL deriv2[MAXCOORD][MAXCOORD];
               } *e_curve; 

double total_sqcurve;  /* total square curvature integral */
double total_length;

/***********************************************************************
*
*  Function: sqcurve_energy_init()
*
*  Purpose: Initializes data structures for square curvature.
*           Call before doing facet loop in calc_energy or calc_force.
*/

void sqcurve_energy_init()
{
  total_sqcurve = 0.0;
  total_length  = 0.0;
  if ( web.dimension != SOAPFILM ) return; /* not needed for string */

  v_curve = (struct v_curve *)temp_calloc(web.skel[VERTEX].max_ord+1,
                   sizeof(struct v_curve));
  if ( !v_curve )
      error("Cannot allocate memory for square curvature.\n",RECOVERABLE);
}

/***********************************************************************
*
*  Function: sqcurve_force_init()
*
*  Purpose: Initializes data structures for square curvature.
*           Call before doing facet loop in calc_energy or calc_force.
*/

void sqcurve_force_init()
{
  if ( web.dimension != SOAPFILM ) return; /* not needed for string */

  v_curve = (struct v_curve *)temp_calloc(web.skel[VERTEX].max_ord+1,
                   sizeof(struct v_curve));
  e_curve = (struct e_curve *)temp_calloc(web.skel[EDGE].max_ord+1,
                   sizeof(struct e_curve));
  if ( !v_curve || !e_curve )
      error("Cannot allocate memory for square curvature.\n",RECOVERABLE);
}

/********************************************************************
*
*  Function: sqcurve_energy()
*
*  Purpose:  Does square curvature energy calculation for a facet.
*
*/

void sqcurve_energy(v_id,side)
vertex_id *v_id;  /* vertex list for facet */
REAL (*side)[MAXCOORD];  /* side vectors */
{ 
  double t1t1,t1t2,t2t2;
  double det;
  struct v_curve *vc[FACET_VERTS];
  int i;
  double area;

  t1t1 = dot(side[0],side[0],web.sdim);
  t1t2 = dot(side[0],side[1],web.sdim);
  t2t2 = dot(side[1],side[1],web.sdim);

  det = t1t1*t2t2 - t1t2*t1t2;

  area = sqrt(det)/2;
  for ( i = 0 ; i < FACET_VERTS ; i++ )
    { vc[i] = v_curve + ordinal(v_id[i]);
      vc[i]->area += area;
    }

  for ( i = 0 ; i < web.sdim ; i++ )
   { vc[0]->force[i] -= (t2t2*side[0][i]-t1t2*side[1][i])/4/area;
     vc[1]->force[i] += (t2t2*side[0][i]-t1t2*side[1][i])/4/area;
     vc[2]->force[i] += (t1t1*side[1][i]-t1t2*side[0][i])/4/area;
     vc[1]->force[i] -= (t1t1*side[1][i]-t1t2*side[0][i])/4/area;
   }
}


/************************************************************************
*
*  Function: sqcurve_force()
*
*  Purpose:  Does square curvature force calculation for a facet.
*
*/

void sqcurve_force(v_id,e_id,side)
vertex_id *v_id;  /* vertex list for facet */
edge_id *e_id;    /* edge list */
REAL (*side)[MAXCOORD];  /* side vectors */
{ 
  double det;
  struct v_curve *vc[FACET_VERTS];
  int i,j,k;
  REAL force[FACET_VERTS][MAXCOORD];
  double tt[FACET_VERTS][FACET_VERTS]; /* side dot products */
  double area;
  struct e_curve *ec[FACET_EDGES];

  for ( j = 0 ; j < FACET_VERTS ; j++ )
    for ( k = 0 ; k <= j ; k++ )
      tt[j][k] = tt[k][j] = dot(side[j],side[k],web.sdim);

  det = tt[0][0]*tt[1][1] - tt[0][1]*tt[0][1];

  area = sqrt(det)/2;
  for ( i = 0 ; i < FACET_VERTS ; i++ )
    { vc[i] = v_curve + ordinal(v_id[i]);
      vc[i]->area += area;
    }
  for ( i = 0 ; i < FACET_EDGES ; i++ )
    ec[i] = e_curve + ordinal(e_id[i]);

  memset((char*)force,0,sizeof(force));
  for ( j = 0 ; j < FACET_VERTS ; j++ )
    { int i1 = (j+1)%FACET_VERTS;
      int i2 = (j+2)%FACET_VERTS;
      for ( i = 0 ; i < web.sdim ; i++ )
        force[j][i] = (tt[i1][i1]*side[i2][i] - tt[i1][i2]*side[i1][i])
                           /4/area;
    }

  /* first and second derivatives at vertices */
  for ( i = 0 ; i < FACET_VERTS ; i++ )
    { int ii = (i+1)%FACET_VERTS; /* opposite side from vertex i */
      for ( j = 0 ; j < web.sdim ; j++ )
        vc[i]->force[j] += force[i][j];
      for ( j = 0 ; j < web.sdim ; j++ )
        { vc[i]->deriv2[j][j] +=  tt[ii][ii]/4/area;
          for ( k = 0 ; k < web.sdim ; k++ )
            vc[i]->deriv2[j][k] -= (force[i][j]*force[i][k]+0.25*side[ii][j]
                                    *side[ii][k])/area;
        }
    }

  /* now first and second derivatives on edges */
  for ( i = 0 ; i < FACET_EDGES ; i++ )
    { 
      int i1 =  (i+1)%FACET_EDGES;
      int i2 =  (i+2)%FACET_EDGES;
        for ( j = 0 ; j < web.sdim ; j++ )
          { ec[i]->deriv2[j][j] += tt[i1][i2]/4/area;
            if ( inverted(e_id[i]) )
              { ec[i]->deriv[0][j] += force[i1][j];
                ec[i]->deriv[1][j] += force[i][j];
              }
            else
              { ec[i]->deriv[0][j] += force[i][j];
                ec[i]->deriv[1][j] += force[i1][j];
              }
            for ( k = 0 ; k < web.sdim ; k++ )
              if ( inverted(e_id[i]) )
                ec[i]->deriv2[k][j] += (-side[i1][j]*side[i2][k]/2
                                   + side[i2][j]*side[i1][k]/4
                                        -force[i1][j]*force[i][k])/area;
              else
                ec[i]->deriv2[j][k] += (-side[i1][j]*side[i2][k]/2
                                   + side[i2][j]*side[i1][k]/4
                                        -force[i1][j]*force[i][k])/area;
          }
     } 
 }

/*************************************************************************
*
*  function: sqcurve_energy_end()
*
*  purpose:  Convert square curvature data into energy.
*            Call at end of facet loop in calc_energy.
*
*/

void sqcurve_energy_end()
{
  vertex_id v_id;
  double modulus = 3*web.params[square_curvature_param].value;
  double energy = 0.0;


  if ( web.dimension == STRING )
    { web.total_energy += total_sqcurve*total_length;
      return; /* rest not needed for string */
    }

  FOR_ALL_VERTICES(v_id)
   { REAL *force;
     double venergy;
     ATTR attr = get_vattr(v_id);


     if ( attr & (FIXED|BOUNDARY) ) continue;
     force = v_curve[ordinal(v_id)].force;
     if ( attr & CONSTRAINT )
       {
          MAP conmap = get_v_constraint_map(v_id);
          int j,oncount = 0;
          struct constraint *con[CONSTRMAX];
          int conlist[CONSTRMAX];
          REAL perp[MAXCOORD];

          for ( j = 0 ; j < web.concount ; j++,conmap>>=1 )
            { if ( !(conmap & 1) ) continue;
              if ( get_v_constraint_status(v_id,j) == ON_CONSTRAINT )
                { conlist[oncount] = j;
                  con[oncount++] = get_constraint(j);
                }
            }

          constr_proj(TANGPROJ,oncount,con,get_coord(v_id),
                         force,perp,NULL,NO_DETECT);
          for ( j = 0 ; j < web.sdim ; j++ )
            force[j] -= perp[j];
       }
     venergy = dot(force,force,web.sdim)/v_curve[ordinal(v_id)].area;
     energy += venergy;

   }
  web.total_energy += modulus*energy/4; /* /4 for (k1+k2)/2 */

  temp_free((char*)v_curve);  v_curve = NULL;
}

/*************************************************************************
*
*  function: sqcurve_force_end()
*
*  purpose:  Convert square curvature data into forces.
*
*/

void sqcurve_force_end()
{
  vertex_id v_id;
  edge_id e_id;
  int i,j;
  double modulus = 0.75*web.params[square_curvature_param].value;
  double e,e1,e2;

  if ( web.dimension != SOAPFILM )
    { sqcurve_force_string_end();
      return;
    }

  FOR_ALL_VERTICES(v_id)
    { REAL *force = get_force(v_id);
      struct v_curve *vc = v_curve + ordinal(v_id);
      ATTR attr = get_vattr(v_id);

      if ( attr & (FIXED|BOUNDARY) ) continue;
      if ( attr & CONSTRAINT )
       {
          MAP conmap = get_v_constraint_map(v_id);
          int j,oncount = 0;
          struct constraint *con[CONSTRMAX];
          int conlist[CONSTRMAX];
          REAL perp[MAXCOORD];

          for ( j = 0 ; j < web.concount ; j++,conmap>>=1 )
            { if ( !(conmap & 1) ) continue;
              if ( get_v_constraint_status(v_id,j) == ON_CONSTRAINT )
                { conlist[oncount] = j;
                  con[oncount++] = get_constraint(j);
                }
            }

          constr_proj(TANGPROJ,oncount,con,get_coord(v_id),
                         vc->force,perp,NULL,NO_DETECT);
          for ( j = 0 ; j < web.sdim ; j++ )
            vc->force[j] -= perp[j];
       }
      e = dot(vc->force,vc->force,web.sdim)/vc->area;
      for ( i = 0 ; i < web.sdim ; i++ )
        force[i] -= modulus*(2*dot(vc->force,vc->deriv2[i],web.sdim)
              - e*vc->force[i])/vc->area;
    }

  FOR_ALL_EDGES(e_id)
    {
      vertex_id headv = get_edge_headv(e_id);
      vertex_id tailv = get_edge_tailv(e_id);
      struct e_curve *ec = e_curve + ordinal(e_id);
      struct v_curve *vc1 = v_curve + ordinal(tailv);
      struct v_curve *vc2 = v_curve + ordinal(headv);
      REAL *force1 = get_force(tailv); 
      REAL *force2 = get_force(headv); 
      
      e1 = dot(vc1->force,vc1->force,web.sdim)/vc1->area;
      e2 = dot(vc2->force,vc2->force,web.sdim)/vc2->area;
      if ( !(get_vattr(tailv) & (FIXED|BOUNDARY)) )
       for ( i = 0 ; i < web.sdim ; i++ )
        { /* force on head due to curvature at tail */
          force2[i] -= modulus*(- e1*ec->deriv[1][i]
            + 2*dot(vc1->force,ec->deriv2[i],web.sdim))/vc1->area;
        }
      if ( !(get_vattr(headv) & (FIXED|BOUNDARY)) )
        for ( i = 0 ; i < web.sdim ; i++ )
        { /* force on tail due to curvature at head */
          force1[i] -= modulus*(-e2*ec->deriv[0][i]/vc2->area);
          for ( j = 0 ; j < web.sdim ; j++ )
            force1[i] -= modulus*2*vc2->force[j]*ec->deriv2[j][i]/vc2->area;
        }
     }

  temp_free((char*)v_curve);  v_curve = NULL;
  temp_free((char*)e_curve);  e_curve = NULL;
}

/************************************************************************
*
*  function: sqcurve_energy_string()
*
*  purpose:  Calculate square curvature energy for string model
*            Works locally vertex by vertex. Requires edges to be
*            on a facet, assumes two edges per vertex.
*
*/

void sqcurve_energy_string(v_id)
vertex_id v_id;
{
  double s1,s2,s1s2;  /* edge lengths */
  REAL side1[MAXCOORD],side2[MAXCOORD];
  facetedge_id fe_id = get_vertex_fe(v_id);
  edge_id e1;
  edge_id e2;
  double energy;

  if ( !valid_id(fe_id) )
    error("Edges must be on a facet for square curvature!\n",RECOVERABLE);

  e2 = get_fe_edge(fe_id);
  if ( !equal_id(v_id,get_edge_tailv(e2)) )
    { e2 = inverse_id(e2);
      fe_id = inverse_id(fe_id);
    }
  e1 = get_fe_edge(get_prev_edge(fe_id));

  get_edge_side(e1,side1);
  get_edge_side(e2,side2);
  s1 = get_edge_length(e1);
  s2 = get_edge_length(e2);
  s1s2 = dot(side1,side2,web.sdim);

  energy = (1 - s1s2/s1/s2)/(s1 + s2);

  energy *= web.params[square_curvature_param].value;

  /* fudge factor to agree with def */
  total_sqcurve += 4*energy;
  total_length += (s1+s2)/2;

}

/************************************************************************
*
*  function: sqcurve_force_string()
*
*  purpose:  Calculate square curvature force for string model
*            Works locally vertex by vertex. Requires edges to be
*            on a facet, assumes two edges per vertex.
*
*/

void sqcurve_force_string(v_id)
vertex_id v_id;
{
  double s1,s2,s1s2;  /* edge lengths */
  REAL side1[MAXCOORD],side2[MAXCOORD];
  facetedge_id fe_id = get_vertex_fe(v_id);
  edge_id e1;
  edge_id e2;
  REAL *tforce,*vforce,*hforce;
  int  i;
  double c1,c2;
  REAL *r;

  if ( !valid_id(fe_id) )
    error("Edges must be on a facet for square curvature!\n",RECOVERABLE);

  e2 = get_fe_edge(fe_id);
  if ( !equal_id(v_id,get_edge_tailv(e2)) )
    { e2 = inverse_id(e2);
      fe_id = inverse_id(fe_id);
    }
  e1 = get_fe_edge(get_prev_edge(fe_id));
  if ( !equal_id(v_id,get_edge_tailv(e2)) )
    error("e2 edge direction confusion!\n",RECOVERABLE);
  if ( !equal_id(v_id,get_edge_headv(e1)) )
    error("e1 edge direction confusion!\n",RECOVERABLE);
 
  vforce = get_force(v_id);
  tforce = get_force(get_edge_tailv(e1));
  hforce = get_force(get_edge_headv(e2));

  get_edge_side(e1,side1);
  get_edge_side(e2,side2);
  s1 = get_edge_length(e1);
  s2 = get_edge_length(e2);
  s1s2 = dot(side1,side2,web.sdim);

  /* save length gradient for projection */
  r = get_restore(v_id);
  for ( i = 0 ; i < web.sdim ; i++ )
    r[i] = side1[i]/s1 - side2[i]/s2;

  c1 = 4*(1 - s1s2*(2*s1+s2)/s2/s1/s1)/s1/(s1+s2)/(s1+s2);
  c2 = 4/s1/s2/(s1+s2);
  for ( i = 0 ; i < web.sdim ; i++ )
    { tforce[i] -= c1*side1[i] + c2*side2[i];
      vforce[i] += c1*side1[i] + c2*side2[i];
    }
  c1 = 4*(1 - s1s2*(2*s2+s1)/s1/s2/s2)/s2/(s1+s2)/(s1+s2);
  c2 = 4/s1/s2/(s1+s2);
  for ( i = 0 ; i < web.sdim ; i++ )
    { hforce[i] += c2*side1[i] + c1*side2[i];
      vforce[i] -= c2*side1[i] + c1*side2[i];
    }
}

/*************************************************************
*
*  function: sqcurve_force_string_end()
*
*  purpose: project force onto length constraint
*
*/

void sqcurve_force_string_end()
{
  vertex_id v_id;
  double lambda = 0.0;  /* correction factor */
  double alpha = 0.0;    /* length restoring factor */
  double fgsum = 0.0;  /* energy dot constraint grad */
  double ggsum = 0.0;  /* constraint grad dot constraint grad */
  int i;
  REAL *r,*f;

  FOR_ALL_VERTICES(v_id)
    { r = get_restore(v_id);
      f = get_force(v_id);
      fgsum += dot(f,r,web.sdim);
      ggsum += dot(r,r,web.sdim);
    }
  
  if ( ggsum != 0.0 )
   { lambda = fgsum/ggsum; 
     alpha = (target_length - web.total_area)/ggsum;
   }

  FOR_ALL_VERTICES(v_id)
    { r = get_restore(v_id);
      f = get_force(v_id);
      for ( i = 0 ; i < web.sdim ; i++ )
        { 
#ifdef PLAIN_SQ
          f[i] -= lambda*r[i];
          r[i] *= alpha;  /* want a restoring motion */
#else
          /* minimizing length*sqcurve */
          f[i] = f[i]*total_length - total_sqcurve*r[i];
          r[i] = 0.0;
#endif
        }
    }
}


