/*
$Header: c:/tads/tads2/RCS/TOK.H 1.6 94/11/06 13:07:42 mroberts Exp $
*/

/* Copyright (c) 1991 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  tok.h - tokenizer definitions
Function
  Definitions for tokenizer
Notes
  None
Modified
  08/14/91 MJRoberts     - creation
*/

#ifndef TOK_INCLUDED
#define TOK_INCLUDED

#ifndef STD_INCLUDED
#include "std.h"
#endif
#ifndef LIN_INCLUDED
#include "lin.h"
#endif
#ifndef MCH_INCLUDED
#include "mch.h"
#endif
#ifndef ERR_INCLUDED
#include "err.h"
#endif
#ifndef MCM_INCLUDED
#include "mcm.h"
#endif

/* number of entries in hash table - must be power of 2 */
#define TOKHASHSIZE 256

/* symbol types */
#define TOKSTUNK      0                  /* unknown symbol, not yet defined */
#define TOKSTFUNC     1                 /* function; value is object number */
#define TOKSTOBJ      2                   /* object; value is object number */
#define TOKSTPROP     3               /* property; value is property number */
#define TOKSTLOCAL    4             /* a local variable or formal parameter */
#define TOKSTSELF     5                         /* the pseudo-object "self" */
#define TOKSTBIFN     6                              /* a built-in function */
#define TOKSTFWDOBJ   7                        /* forward-referenced object */
#define TOKSTFWDFN    8                        /* forward-referenced object */
#define TOKSTINHERIT  9                    /* the pseudo-object "inherited" */
#define TOKSTEXTERN  10                             /* an external function */
#define TOKSTKW      11                   /* keyword; value is token number */
#define TOKSTLABEL   12                                  /* statement label */
#define TOKSTARGC    13                       /* 'argcount' pseudo-variable */

/* token types */
#define TOKTEOF     1

/* binary operators - keep these together (see prsbopl[] in prs.c) */
#define TOKTPLUS    2
#define TOKTMINUS   3
#define TOKTDIV     4
#define TOKTTIMES   5
#define TOKTNOT     6                                         /* ! or "not" */
#define TOKTEQ      7
#define TOKTNE      8
#define TOKTGT      9
#define TOKTGE      10
#define TOKTLT      11
#define TOKTLE      12
#define TOKTMOD     13
#define TOKTBAND    14
#define TOKTBOR     15
#define TOKTXOR     16
#define TOKTSHL     17
#define TOKTSHR     18
#define TOKTTILDE   30

#define TOKTLPAR    50                                                 /* ( */
#define TOKTRPAR    51                                                 /* ) */
#define TOKTCOLON   52
#define TOKTDSTRING 53                           /* string in double quotes */
#define TOKTSSTRING 54                           /* string in single quotes */
#define TOKTNUMBER  55
#define TOKTSYMBOL  56
#define TOKTINVALID 57                             /* invalid lexical token */
#define TOKTLBRACK  58                                                 /* [ */
#define TOKTRBRACK  59                                                 /* ] */
#define TOKTLBRACE  60                                                 /* { */
#define TOKTRBRACE  61                                                 /* } */
#define TOKTSEM     62                                                 /* ; */
#define TOKTCOMMA   63
#define TOKTDOT     64                                                 /* . */
#define TOKTOR      65                                         /* | or "if" */
#define TOKTAND     66                                        /* & or "and" */
#define TOKTIF      67                                          /* keywords */
#define TOKTELSE    68
#define TOKTWHILE   69
#define TOKTFUNCTION 70
#define TOKTRETURN  71
#define TOKTLOCAL   72
#define TOKTOBJECT  73
#define TOKTBREAK   74
#define TOKTCONTINUE 75
#define TOKTLIST    76                                            /* a list */
#define TOKTNIL     77
#define TOKTTRUE    78
#define TOKTPASS    79
#define TOKTCLASS   80
#define TOKTEXIT    81
#define TOKTABORT   82
#define TOKTASKDO   83
#define TOKTASKIO   84
#define TOKTPOUND   85                                                 /* # */
#define TOKTQUESTION 86                                                /* ? */
#define TOKTCOMPOUND 87
#define TOKTIOSYN   88
#define TOKTDOSYN   89
#define TOKTEXTERN  90
#define TOKTFORMAT  91
#define TOKTDO      92
#define TOKTFOR     93
#define TOKTNEW     94
#define TOKTDELETE  95

/* assignment operators - keep these together */
#define TOKTINC     150                                               /* ++ */
#define TOKTPOSTINC 151                              /* MUST BE TOKTINC + 1 */
#define TOKTDEC     152                                               /* -- */
#define TOKTPOSTDEC 153                              /* MUST BE TOKTDEC + 1 */
#define TOKTPLEQ    154                                               /* += */
#define TOKTMINEQ   155                                               /* -= */
#define TOKTDIVEQ   156                                               /* /= */
#define TOKTTIMEQ   157                                               /* *= */
#define TOKTASSIGN  158                                /* simple assignment */
#define TOKTMODEQ   159                     /* %= (mod and assign) operator */
#define TOKTBANDEQ  160                                               /* &= */
#define TOKTBOREQ   161                                               /* |= */
#define TOKTXOREQ   162                              /* ^= (xor and assign) */
#define TOKTSHLEQ   163                      /* <<= (shift left and assign) */
#define TOKTSHREQ   164                      /* >>= (shift right and assign */

#define TOKTSWITCH  200
#define TOKTCASE    201
#define TOKTDEFAULT 202
#define TOKTGOTO    203
#define TOKTELLIPSIS 204                                             /* ... */
#define TOKTSPECIAL 205                                   /* "specialWords" */
#define TOKTREPLACE 206                                          /* replace */
#define TOKTMODIFY  207                                           /* modify */

#define TOKTEQEQ    208                                /* the '==' operator */
#define TOKTPOINTER 209                                  /* the -> operator */

/* the longest a symbol name can be */
#define TOKNAMMAX 39

/* symbol table entry */
typedef struct toksdef toksdef;
struct toksdef
{
    uchar    tokstyp;                                 /* type of the symbol */
    uchar    tokshsh;                               /* hash value of symbol */
    ushort   toksval;              /* value of the symbol (depends on type) */
    ushort   toksfr;               /* frame offset of symbol (for debugger) */
    uchar    tokslen;                        /* length of the symbol's name */
    char     toksnam[TOKNAMMAX];                          /* name of symbol */
};

/* symbol table entry without 'name' portion - for allocation purposes */
struct toks1def
{
    uchar    tokstyp;
    uchar    tokshsh;
    ushort   toksval;
    ushort   toksfr;
    uchar    tokslen;
    char     toksnam[1];
};
typedef struct toks1def toks1def;

/* generic symbol table object - other symbol tables are subclasses */
typedef struct toktdef toktdef;
struct toktdef
{
    void     (*toktfadd)(/*_ toktdef *tab, char *name, int namel, int typ,
                             int val, int hash _*/);          /* add symbol */
    int      (*toktfsea)(/*_ toktdef *tab, char *name, int namel, int hash,
                             toksdef *ret _*/);      /* search symbol table */
    void     (*toktfset)(/*_ toktdef *tab, toksdef *sym _*/);
                             /* update val & typ of symbol to those in *sym */
    void     (*toktfeach)(/*_ toktdef *tab,
                              void (*fn)(dvoid *ctx, toksdef *sym),
			      dvoid *fnctx _*/);    /* call fn for each sym */
    toktdef   *toktnxt;                 /* next symbol table to be searched */
    errcxdef  *tokterr;                           /* error handling context */
};

/* maximum number of pools (TOKTSIZE bytes each) for symbols */
#define TOKPOOLMAX 128

/* pointer to a symbol in a hashed symbol table */
struct tokthpdef
{
    mcmon  tokthpobj;                /* cache manager object number of page */
    uint   tokthpofs;                  /* offset within page of this symbol */
};
typedef struct tokthpdef tokthpdef;

/* extended symbol entry in a hashed symbol table */
struct tokshdef
{
    tokthpdef tokshnxt;              /* pointer to next symbol in the table */
    toksdef   tokshsc;                  /* superclass - normal symbol entry */
};
typedef struct tokshdef tokshdef;

/* hashing symbol table (subclass of generic symbol table) */
struct tokthdef
{
    toktdef   tokthsc;              /* generic symbol table superclass data */
    mcmcxdef *tokthmem;                           /* memory manager context */
    tokthpdef tokthhsh[TOKHASHSIZE];                          /* hash table */
    uint      tokthpcnt;            /* number of memory pools for toksdef's */
    mcmon     tokthpool[TOKPOOLMAX];          /* memory pools for toksdef's */
    uint      tokthfinal[TOKPOOLMAX];        /* actual sizes of these pools */
    uchar    *tokthcpool;                           /* current pool pointer */
    ushort    tokthsize;               /* remaining size of top memory pool */
    ushort    tokthofs;             /* allocation offset in top memory pool */
};
typedef struct tokthdef tokthdef;

/* size of toksdef pools to allocate for hashed symbol tables */
#define TOKTHSIZE 4096

/*
 *   Linear cache-object-embedded symbol table.  This type of symbol
 *   table is used for frame parameter/local variable lists.  It is best
 *   for small tables, because it isn't broken up into hash buckets, so it
 *   is searched linearly.  As a result, it's small enough to be embedded
 *   in code.
 */
struct toktldef
{
    toktdef   toktlsc;              /* generic symbol table superclass data */
    uchar    *toktlptr;                      /* base of linear symbol table */
    uchar    *toktlnxt;                          /* next free byte in table */
    uint      toktlcnt;                   /* number of objects in the table */
    uint      toktlsiz;                     /* bytes remaining in the table */
};
typedef struct toktldef toktldef;

struct tokdef
{
    int	     toktyp;                                   /* type of the token */
    int      toklen;           /* length of token text, if a symbolic token */
    long     tokval;                        /* numeric value, if applicable */
    ushort   tokofs;
    uint     tokhash;              /* token hash value, if a symbolic token */
    char     toknam[TOKNAMMAX+1];     /* text of token, if a symbolic token */
    toksdef  toksym;                    /* symbol from table matching token */
};
typedef struct tokdef tokdef;

/* special character sequence */
#define TOKSCMAX  3            /* maximum length of a special char sequence */
typedef struct tokscdef tokscdef;
struct tokscdef
{
    tokscdef *tokscnxt;          /* next sequence with same first character */
    int       toksctyp;             /* token type corresponding to sequence */
    int       toksclen;                           /* length of the sequence */
    char      tokscstr[TOKSCMAX+1];                  /* the sequence itself */
};

/*
 *   Compare a special character sequence - for efficiency, define
 *   something special for the maximum length available (TOKSCMAX).
 *   Note that the first character will always be equal, or the
 *   string wouldn't even get to the point of being tested by this
 *   macro.
 */
#if TOKSCMAX == 3
# define toksceq(str1, str2, len1, len2) \
 ((len2) >= (len1) && (len1) == 1 || ((str1)[1] == (str2)[1] && \
 ((len1) == 2 || (str1)[2] == (str2)[2])))
#endif /* TOKSCMAX == 3 */
#ifndef toksceq
# define toksceq(str1, str2, len) (!memcmp(str1, str2, (size_t)(len)))
#endif /* toksceq */

/* special character sequence list table entry */
struct tokldef
{
    int  tokltyp;                   /* token type corresponding to sequence */
    char toklstr[TOKSCMAX+1];                   /* the text of the sequence */
};
typedef struct tokldef tokldef;

/* include path structure */
typedef struct tokpdef tokpdef;
struct tokpdef
{
    tokpdef *tokpnxt;                                  /* next path in list */
    int      tokplen;                           /* length of directory name */
    char     tokpdir[1];                             /* directory to search */
};

/* #define symbol structure */
typedef struct tokdfdef tokdfdef;
struct tokdfdef
{
    tokdfdef *nxt;                    /* next symbol in the same hash chain */
    char     *nm;                                     /* name of the symbol */
    int       len;                                  /* length of the symbol */
    int       explen;                            /* length of the expansion */
    char      expan[1];                                 /* expansion buffer */
};

/* #define hash table information */
#define TOKDFHSHSIZ   64
#define TOKDFHSHMASK  63

/* maximum #if nesting */
#define TOKIFNEST     64

/* #if state */
#define TOKIF_IF_YES    1             /* processing a true #if/#ifdef block */
#define TOKIF_IF_NO     2            /* processing a false #if/#ifdef block */
#define TOKIF_ELSE_YES  3                   /* processing a true #else part */
#define TOKIF_ELSE_NO   4                  /* processing a false #else part */

/* maximum macro expansion nesting */
#define TOKMACNEST  20

/* lexical analysis context */
struct tokcxdef
{
    errcxdef *tokcxerr;                           /* error handling context */
    mcmcxdef *tokcxmem;                            /* cache manager context */
    struct    dbgcxdef *tokcxdbg;                       /* debugger context */
    lindef   *tokcxlin;                                      /* line source */
    tokpdef  *tokcxinc;                        /* head of include path list */
    toktdef  *tokcxstab;              /* current head of symbol table chain */
    dvoid    *tokcxscx;    /* context for string storage callback functions */
    ushort  (*tokcxsst)(/*_ dvoid *ctx _*/);
               /* start storing a string; return offset of string's storage */
    void    (*tokcxsad)(/*_ dvoid *ctx, char *str, ushort len _*/);
                                              /* add characters to a string */
    void    (*tokcxsend)(/*_ dvoid *ctx _*/);      /* finish storing string */
    char     *tokcxmsav[TOKMACNEST]; /* saved positions for macro expansion */
    ushort    tokcxmsvl[TOKMACNEST];   /* saved lengths for macro expansion */
    int       tokcxmlvl;                             /* macro nesting level */
    int       tokcxflg;                                            /* flags */
#   define    TOKCXFINMAC    0x01         /* doing <<expr>> macro expansion */
#   define    TOKCXCASEFOLD  0x02              /* fold upper and lower case */
#   define    TOKCXFCMODE    0x04                /* parse using C operators */
    tokdef    tokcxcur;                                    /* current token */
    char     *tokcxbuf;                            /* buffer for long lines */
    ushort    tokcxbsz;                         /* size of long line buffer */
    char     *tokcxptr;                         /* pointer into line source */
    ushort    tokcxlen;                         /* length of text in buffer */
    uchar     tokcxinx[256];                   /* special character indices */
    tokdfdef *tokcxdf[TOKDFHSHSIZ];       /* hash table for #define symbols */
    int       tokcxifcnt;           /* number of #endif's we expect to find */
    char      tokcxif[TOKIFNEST];       /* #if state for each nesting level */
    struct    linfdef  *tokcxhdr;    /* list of previously included headers */
    tokscdef *tokcxsc[1];                        /* special character table */
};
typedef struct tokcxdef tokcxdef;


/* allocate and initialize a lexical analysis context */
tokcxdef *tokcxini(/*_ errcxdef *errctx, mcmcxdef *mctx, tokldef *sctab _*/);

/* add an include path to a token handling context */
void tokaddinc(/*_ tokcxdef *ctx, char *path, int pathlen _*/);

/* compute the hash value of a string */
uint tokhsh(/*_ char *nam _*/);

/* initialize a hashed symbol table */
void tokthini(/*_ errcxdef *ctx, toktdef *toktab _*/);

/* add a symbol to a hashed symbol table */
void tokthadd(/*_ toktdef *toktab, char *name, int namel,
                  int typ, int val, int hash _*/);

/* update a symbol in a hashed symbol table */
void tokthset(/*_ toktdef *toktab, toksdef *sym _*/);

/* search a hashed symbol table for a symbol */
int tokthsea(/*_ toktdef *tab, char *name, int namel, int hash,
                 toksdef *ret _*/);

/* call a function for each symbol in a hashed symbol table */
void toktheach(/*_ toktdef *tab, void (*cb)(dvoid *ctx, toksdef *sym),
                   dvoid *ctx _*/);

/* find a symbol given type and value */
int tokthfind(/*_ toktdef *tab, int typ, uint val, toksdef *sym _*/);

/* initialize a linear symbol table */
void toktlini(/*_ errcxdef *errctx, toktldef *toktab,
                  uchar *mem, uint siz _*/);

/* add a symbol to a linear symbol table */
void toktladd(/*_ toktdef *toktab, char *name, int namel,
                  int typ, int val, int hash _*/);
		
/* search a linear symbol table */
int toktlsea(/*_ toktdef *tab, char *name, int namel, int hash,
                 toksdef *ret _*/);

/* update a symbol in a linear symbol table */
void toktlset(/*_ toktdef *toktab, toksdef *sym _*/);

/* call a function for each symbol in a local symbol table */
void toktleach(/*_ toktdef *tab, void (*cb)(dvoid *ctx, toksdef *sym),
                   dvoid *ctx _*/);

/* delete all symbols from a linear table */
void toktldel(/*_ toktldef *tab _*/);

/* get next token, removing it from input stream */
int toknext(/*_ tokcxdef *ctx _*/);

/* general function to get/peek at next token */
int tokget1(/*_ tokcxdef *ctx, tokdef *tok, int consume _*/);

/* determine if a char is a valid non-initial character in a symbol name */
#define TOKISSYM(c) (isalpha(c) || isdigit(c) || (c)=='_' || (c)=='$')

/* numeric conversion and checking macros */
#define TOKISHEX(c) (isdigit(c)||((c)>='a'&&(c)<='f')||((c)>='A'&&(c)<='F'))
#define TOKISOCT(c) (isdigit(c)&&!((c)=='8'||(c)=='9'))

#define TOKHEX2INT(c) (isdigit(c)?(c)-'0':((c)>='a'?(c)-'a'+10:(c)-'A'+10))
#define TOKOCT2INT(c) ((c)-'0')
#define TOKDEC2INT(c) ((c)-'0')

#endif /* TOK_INCLUDED */
