/* garbage management */

/*
 * Copyright 1989 Jonathan Lee.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */

#include "fools.h"

#ifndef lint
static char SccsId[] = "@(#)gc.c	1.6 1/11/90";
#endif

#ifndef USE_MALLOC
static Mem bigAlloc, smallAlloc;
#endif /* USE_MALLOC */

#ifdef DEBUG

/* Info about objects in use. */
static void gcStats()
{
    if (debugGC) {
#ifndef USE_MALLOC
	(void)printf("%d (%d/%d) objects in use\n",
		     bigAlloc->count + smallAlloc->count,
		     bigAlloc->count, smallAlloc->count);
#endif /* !defined(USE_MALLOC) */
	(void)printf("%d list nodes\n", listAlloc->count);
	(void)printf("%d tree nodes\n", treeAlloc->count);
    }
}

#endif /* DEBUG */

/* sizes for allocating blocks */
#define BIGGEST	(sizeof (frameInst_t))
#define MEDIUM	(sizeof (pairInst_t))

/* initialize the garbage collector */
void gcInit()
{
#ifndef USE_MALLOC
    bigAlloc = memNew(ALIGNSIZE(BIGGEST), 256);
    smallAlloc = memNew(ALIGNSIZE(MEDIUM), 1024);
#endif /* USE_MALLOC */
}

/* callback to free a list. */
static void gcCallback(list)
     List list;
{
    listFree(list, (F_VOID)NULL);
}

/* Return a list that can be used locally and is freed by gcListFree or
 * an error.  */
List gcListNew(cb)
     Callback_t *cb;
{
    List new;

    cb->arg = (Ptr)(new = listNew());
    cb->func = gcCallback;

    errorPushCB(cb);
    return new;
}

/* Free the list allocated by gcListNew.  Since callbacks are placed on
 * a stack, the lists must be free'd in the reverse order in which they
 * were created. */
void gcListFree(list)
     List list;
{
    errorPopCB();
    gcCallback(list);
}

/* Begin a gc level in which all objects that are allocated during this
 * period and remain unused can be destroyed by gcEnd. It is possible to
 * nest gc levels but not overlap them. */
void gcBegin()
{
    intDisable();
    listPush((Ptr)(gcList = listNew()), gcStack);
}

/* End (free all unused) objects created during this gc level. */
void gcEnd()
{
    intDisable();

#ifdef DEBUG
    gcStats();
    if (debugGC)
	printf("garbage collecting...\n");
#endif

    ASSERT(!listEmpty(gcStack));
    listFree((List)listPop(gcStack), _objUnlink);
    gcList = (List)listPeek(gcStack);

#ifdef DEBUG
    gcStats();
#endif
}

/* Insert object into the current gc level.  Assumes obj has an extra link. */
void gcInsert(obj)
     Obj obj;
{
    ASSERT(gcList != (List)NULL);
    listPush((Ptr)obj, gcList);
}

/* free obj */
void gcFree(obj)
     Obj obj;
{
#ifdef USE_MALLOC
    (void)free((char *)obj);
#else
    memFreeBlock((Ptr)obj, CLASS(obj)->size <= MEDIUM ? smallAlloc : bigAlloc);
#endif /* USE_MALLOC */
}

/* Get a new object that will not be free'd by gcEnd. */
Obj gcNew(class)
     Class class;
{
    Obj new;
    extern char *malloc();

#ifdef USE_MALLOC
    new = (Obj)malloc(class->size); /* largest object */
    bzero(new, class->size);
#else
    ASSERT(class->size <= BIGGEST);
    new	= (Obj)memCleanBlock(class->size <= MEDIUM ? smallAlloc : bigAlloc);
#endif /* USE_MALLOC */

    DATA(new, class, basicInst) = class;
    return new;
}

/* Get a new object that will be free'd by the next gcEnd unless linked. */
Obj gcTemp(class)
     Class class;
{
    Obj new;
    extern char *malloc();

#ifdef USE_MALLOC
    new = (Obj)malloc(class->size); /* largest object */
    bzero(new, class->size);
#else
    ASSERT(class->size <= BIGGEST);
    new	= (Obj)memCleanBlock(class->size <= MEDIUM ? smallAlloc : bigAlloc);
#endif /* USE_MALLOC */

    DATA(new, class, basicInst) = class;
    objLink(new);
    ASSERT(gcList != (List)NULL);
    listPush((Ptr)new, gcList);
    return new;
}
