/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <GrUsrpSink.h>
#include <stdexcept>
#include <config.h>

#ifdef HAVE_USRP

#include <usrp.h>
#include <usrp_config.h>

GrUsrpSink::GrUsrpSink (int which_board,
			unsigned int usrp_tx_config,
			unsigned interp_rate)
  : GrSink (sizeof (VrComplex))
{
  d_ninputs = usrp_tx_config_stream_count (usrp_tx_config);

  d_usrp = usrp_tx::make (which_board);
  if (d_usrp == 0)
    throw std::runtime_error ("can't open usrp");

  fprintf (stderr, "GrUsrpSink: opened!\n");
  
  if (!d_usrp->set_interp_rate (interp_rate))
    throw std::runtime_error ("bad interp_rate");

  // FIXME needs work when we implement usrp_tx_config
}

GrUsrpSink::~GrUsrpSink ()
{
  delete d_usrp;
}

GrUsrpSink *
GrUsrpSink::make (int which_board,
		  unsigned int usrp_tx_config,
		  unsigned interp_rate)
{
  try {
    return new GrUsrpSink (which_board, usrp_tx_config, interp_rate);
  }
  catch (...){
    return 0;
  }
}
		  
void
GrUsrpSink::initialize ()
{
  if (getNumberInputs () != d_ninputs){
    fprintf (stderr, "GrUsrpSink: number of inputs does not match usrp_tx_config\n");
    throw std::runtime_error ("wrong number of inputs");
  }

  // we want to ensure that all calls to d_usrp->write are multiples of 512 bytes.
  // Each VrComplex that we consume results in 2 16-bit shorts being written.
  // Therefore, to meet our constraint:  OutputSize = 512/4 = 128
  
  setOutputSize (128 * 8 * 16);	// make it 4K
}

bool
GrUsrpSink::set_interp_rate (unsigned int rate)
{
  return d_usrp->set_interp_rate (rate);
}

bool
GrUsrpSink::_set_interp_reg (unsigned int regval)
{
  return d_usrp->_set_interp_reg (regval);
}

bool
GrUsrpSink::set_tx_freq (int channel, double freq)
{
  return d_usrp->set_tx_freq (channel, freq);
}

bool
GrUsrpSink::set_ext_clk_div (unsigned int rate)
{
  return d_usrp->set_ext_clk_div (rate);
}

int
GrUsrpSink::work3 (VrSampleRange output,
		   VrSampleRange inputs[], void *ai[])
{
  sync (output.index);		// force in-order execution

  if (getNumberInputs () != d_ninputs){
    throw std::runtime_error ("wrong number of inputs");
  }
  
  VrComplex **in = (VrComplex **) ai;

  static const int BUFSIZE = 8 * 1024;
  short 	outbuf[BUFSIZE];			// 16KB writes
  int		obi = 0;
  unsigned 	n;
  bool		underrun;

  switch (d_ninputs){
  case 1:
    for (n = 0; n < output.size; n++){
      outbuf[obi+0] = (short) real(in[0][n]); // saturate?
      outbuf[obi+1] = (short) imag(in[0][n]); // saturate?
      obi += 2;
      if (obi >= BUFSIZE){		// flush
	if (d_usrp->write (outbuf, obi * sizeof (short), &underrun) != obi * (int) sizeof (short)){
	  // quietly ignore the problem...
	  // throw std::runtime_error ("GrUsrpSink: write");
	}
	if (underrun)
	  fprintf (stderr, "GrUsrpSink: underrun\n");
	
	obi = 0;
      }
    }
    if (obi != 0){
      if (d_usrp->write (outbuf,
			 obi * sizeof (short), &underrun) != (obi * (int) sizeof (short))){
	// quietly ignore the problem...
	// throw std::runtime_error ("GrUsrpSink: write");
      }
      if (underrun)
	fprintf (stderr, "GrUsrpSink: underrun");
    }
    break;
    
  case 2:
  case 3:
  case 4:
  default:
    fprintf (stderr, "GrUsrpSink: only a single input is supported now.\n");
    throw std::runtime_error ("wrong number of inputs");
  }

  return output.size;
}

// ACCESSORS

double
GrUsrpSink::get_oscillator_freq () const
{
  return d_usrp->get_oscillator_freq ();
}

unsigned int
GrUsrpSink::get_interp_rate () const
{
  return d_usrp->get_interp_rate ();
}

double
GrUsrpSink::get_tx_freq (int channel) const
{
  return d_usrp->get_tx_freq (channel);
}

unsigned int
GrUsrpSink::get_ext_clk_div () const
{
  return d_usrp->get_ext_clk_div ();
}

#else

// ----------------------------------------------------------------
//			   stub it all out
// ----------------------------------------------------------------

GrUsrpSink::GrUsrpSink (int which_board,
			unsigned int usrp_tx_config, unsigned interp_rate)
  : GrSink (sizeof (VrComplex))
{
  fprintf (stderr, "GrUsrpSink: stub version.  HAVE_USRP not defined\n");
}

GrUsrpSink::~GrUsrpSink ()
{
}

void
GrUsrpSink::initialize ()
{
}

bool
GrUsrpSink::set_interp_rate (unsigned int rate)
{
  return false;
}

bool
GrUsrpSink::_set_interp_reg (unsigned int regval)
{
  return false;
}

bool
GrUsrpSink::set_tx_freq (int channel, double freq)
{
  return false;
}

bool
GrUsrpSink::set_ext_clk_div (unsigned int rate)
{
  return false;
}

int
GrUsrpSink::work3 (VrSampleRange output,
		   VrSampleRange inputs[], void *ai[])
{
  return output.size;
}

// ACCESSORS

double
GrUsrpSink::get_oscillator_freq () const
{
  return 120e6;
}

unsigned int
GrUsrpSink::get_interp_rate () const
{
  return 256;
}

double
GrUsrpSink::get_tx_freq (int channel) const
{
  return 0;
}

unsigned int
GrUsrpSink::get_ext_clk_div () const
{
  return 12;
}

#endif
