/**********************************************************************

    Copyright (c) 1993, 1994 by Carl Ellison and William Setzer
    This code may be used for any purpose, provided this copyright
    and statement are left intact.

tran.c -- The original code was written by setzer@math.ncsu.edu (William
Setzer) and has been modified by Carl Ellison.

This software is not claimed to offer cryptographic security.  Rather, it
is offered as an illustration of a general cryptographic technique which
has been known and analyzed for years -- transposition of an input source.
See H. F. Gaines "Cryptanalysis", for example, for a discussion of
transposition ciphers.

The particular transposition used here is a self-inverse.  In any block (up
to BLOCKSIZE in length), pairs of input bytes are swapped.  That makes this
one less general than the ones discussed in the literature.

Choice of pairs is made by a PRNG (Pseudo-random number generator).  This
is slightly different from the usage found in textbooks.  Typically, the
transposition key used in those texts is a moderate length word.  The key
here is generated by the PRNG and applies to the whole block.  This then
becomes a columnar transposition with one row and with up to BLOCKSIZE
columns.  Each block of the transposition uses a new key, generated by the
PRNG.  Otherwise, this is the same as the systems described in standard
texts.  The difference is due primarily to the computer.  Prior mechanisms
had to use smaller sizes in order to keep the paperwork down for the human
cipher clerk.  The use of a self-inverse was not necessary but is a
convenience so that the user does not need to specify a direct or inverse
mode when invoking tran.  However, it should be noted that this feature
reduces security of the algorithm, if you were to build a cryptographic
routine along these lines.

The use of the histogram of the first block (possibly the entire file, if
it is less than BLOCKSIZE long) to form the PRNG key is intended to prevent
the kind of cryptanalysis detailed in Gaines: that is, solution in depth.
A common attack on transposition ciphers is to take two messages enciphered
with the same key and lay them side-by-side, anagramming them in sets.
[For more detail, please see Gaines.]  By modifying the key with
information from the file being transposed, using a function which is
invariant under transposition, this program attempts to make it unlikely
that two messages would be found which have the same key, even though there
might easily be two messages transposed using the same password.  Although
a permutation of bytes is a linear operation, the data-dependent
permutation is a non-linear operation and adds strength to the resulting
cipher if used in combination with others (as in des|tran|des|tran|des).

Usage:  as written, tran takes 0 to 2 arguments:

      tran <input file>  <output file>

If either of the files is not specified, stdin or stdout is used
respectively.

**********************************************************************/

#include <stdio.h>

extern void set_rnd_seed();	/* init the PRNG from key[] */
extern unsigned long rnd();	/* get one random integer */

#define BLOCKSIZE (8192)	/* use something larger than 256 */

long perm[BLOCKSIZE];		/* array of byte indices yet to swap */
unsigned char buf[BLOCKSIZE];	/* array of bytes in one block */

FILE *my_fopen(file, type)	/* open a file for R or W and exit if fail */
char *file, *type;		/* file: name; type: r or w */
{
  FILE *fp;			/* resulting file */

  if ( (fp = fopen(file, type))	== NULL ) { /* can't open */
    (void) fprintf(stderr, "Can't open '%s'\n", file); /* print error O'E */
    exit(1);			/* and exit */
  }
  return ( fp ) ;		/* return if success */
} /* end of my_fopen */

int main(argc, argv)		/* get params and do the transposition */
long argc;
char **argv;
{
  register long i, len;
  register FILE *infp, *outfp ;	/* input and output files */
  long savlen, pos ;
  long hist[256] ;		/* histogram of bytes */
  char tmp ;


  infp  = (argc > 1) ? my_fopen(argv[1], "r") : stdin ;	/* input file name */
  outfp = (argc > 2) ? my_fopen(argv[2], "w") : stdout ; /* output file name */

  len = fread(buf, 1, BLOCKSIZE, infp) ;	/* get first block and its length */

  for (i=0 ;i<256 ;i++) hist[i] = 0 ; /* init the histogram array */
  for (i=0 ;i<len ;i++) hist[ buf[i] ]++ ; /* gather the histogram */
  set_rnd_seed(hist) ;		/* init the ranno generator with the hist */

  do {				/* for each block of the input file */
    savlen = len ;		/* remember how long it was */

    for (i = 0; i < len; i++)	/* initialize a permutation array */
      perm[i] = i ;		/* of the same length as the block */
    
#define swap(A,B)  tmp = A; A = B; B = tmp;

    while (len > 1)		/* swap byte pairs as long as there are any */
      {
	pos = 1 + rnd() % (len - 1) ; /* pick a non-0 position */
	swap( buf[perm[0]], buf[perm[pos]] ) ; /* swap two bytes */
				/* replace the two positions swapped */
				/* with ones from the end */
	perm[0]   = perm[(pos == len - 2) ? len - 1 : len - 2] ;
	perm[pos] = perm[len - 1] ;
	len -= 2 ;		/* and reduce the length to be done */
      }
    fwrite(buf, 1, savlen, outfp) ; /* write the transposed block */
  } while ((len = fread(buf, 1, BLOCKSIZE, infp)) > 0) ; /* and read the next one */
  exit(0) ;
} /* end of main() */
