		name	SUPAVGA
		title	Various Super VGA 256-Colour image routines.
		page	132,60

;============================================================================
;
; The bulk of the code to handle different Super VGA boards is
; snarfed from FRACTINT (Version 10).
;
; The procedure for handling these different boards is reasonably
; straightforward.
;
; It appears that all the super VGA modes have the same video
; memory organisation (for 256 colour modes at least), i.e. bank
; switched flat memory pages (usually 64K page size) . The only
; apparent difference between each manufacturer's board is the
; way paging is performed and the INT 10H BIOS initialization values.
; Thus, the only code specific to each board is the paging routine.
;
; To determine the page number of some arbitrary pixel (x, y), use
; the following expression (given a page size of 64K):
;
;		(y * x-width + x) >> 16
;
; After mapping in the video page, the CPU address of the pixel
; will be:
;
;		0xA000: ((y * x-width) & 0xFFFF) + x
;
;
;
; Some boards can handle double buffering for some of their higher
; modes. This is taken advantage of where possible.
;
;============================================================================

		.model	LARGE,c


		extrn	svga_cur_bank:BYTE
		extrn	adapter_x:WORD
		extrn	adapter_y:WORD
		extrn	adapter_bank:DWORD
		extrn	EMS_base:PROC
		extrn	EMS_map:PROC



;--------------------------------------
; Adapter types we think we know about.

ADAPTER_NONE		=	0
ADAPTER_ATI		=	1
ADAPTER_EVEREX		=	2
ADAPTER_TRIDENT		=	3
ADAPTER_VIDEO7		=	4
ADAPTER_PARADISE	=	5
ADAPTER_CHIPS		=	6
ADAPTER_TSENG		=	7
ADAPTER_TSENG4		=	8
ADAPTER_AHEADA		=	9
ADAPTER_AHEADB		=	10
ADAPTER_OTI		=	11
ADAPTER_VGA		=	99


GetEMSByte	MACRO	hdl, pag	;; Get next byte from EMS (remapping afterwards if required)
		lodsb			;; Get byte from DS:SI
		test	si,4000H	;; Check if we've got to the end of the page
		jz	@F
		inc	(pag)		;; Next page number
		push	ax
		mov	ax,(pag)
		push	ax
		mov	ax,(hdl)
		push	ax
		call	EMS_map
		add	sp,4
		pop	ax
		mov	si,base_ofs
@@:
		ENDM

nojmp		macro
		local	lbl
		jmp	lbl
lbl:
		endm


		.code


aheada_bank	proc	
		cli
		mov	SS:svga_cur_bank,al
		push	ax
		push	dx
		push	cx
		mov	ch,al
        	mov     dx,3ceh		;Enable extended registers
        	mov     ax,200fh
        	out     dx,ax
		mov	dl,0cch		;bit 0
		in	al,dx
		mov	dl,0c2h
		and	al,11011111b
		shr	ch,1
		jnc	temp_1
		or	al,00100000b
temp_1:		out	dx,al
		mov	dl,0cfh		;bits 1,2,3
		mov	al,0
		out	dx,al
		inc	dx
		in	al,dx
		dec	dx
		and	al,11111000b
		or	al,ch
		mov	ah,al
		mov	al,0
		out	dx,ax
		sti
		pop	cx
		pop	dx
		pop	ax
		ret
aheada_bank	endp

aheadb_bank	proc	
		cli
		mov	SS:svga_cur_bank,al
		push	ax
		push	dx
		push	cx
		mov	ch,al
        	mov     dx,3ceh		;Enable extended registers
        	mov     ax,200fh
        	out     dx,ax
		mov	ah,ch
		mov	cl,4
		shl	ah,cl
		or	ah,ch
		mov	al,0dh
		out	dx,ax
		sti
		pop	cx
		pop	dx
		pop	ax
		ret
aheadb_bank	endp

ati_bank	proc			;ATI VGA Wonder
		cli
		mov	SS:svga_cur_bank,al
		push	ax
		push	dx
		mov	ah,al
		mov	dx,1ceh
		mov	al,0b2h
		out	dx,al
		inc	dl
		in	al,dx
		shl	ah,1
		and	al,0e1h
		or	ah,al
		mov	al,0b2h
		dec	dl
		out	dx,ax
		sti
		pop	dx
		pop	ax
		ret
ati_bank	endp

chips_bank	proc			;Chips & Tech
		cli
		mov	SS:svga_cur_bank,al
		push	ax
		push	dx
		mov     dx,46e8h	;place chip in setup mode
		mov     ax,1eh
		out     dx,ax
		mov     dx,103h		;enable extended registers
		mov     ax,0080h	; (patched per JB's msg - Bert)
		out     dx,ax
		mov     dx,46e8h	;bring chip out of setup mode
		mov     ax,0eh
		out     dx,ax
		mov	ah,SS:svga_cur_bank
		shl	ah,1		;change 64k bank number into 16k bank number
		shl	ah,1
		mov	al,10h
		mov	dx,3d6h
		out	dx,ax
		sti
		pop	dx
		pop	ax
		ret
chips_bank	endp

everex_bank	proc			;Everex
		cli
		mov	SS:svga_cur_bank,al
		push	ax
		push	dx
		push	cx
		mov	cl,al
		mov	dx,3c4h
		mov	al,8
		out	dx,al
		inc	dl
		in	al,dx
		dec	dl
		shl	al,1
		shr	cl,1
		rcr	al,1
		mov	ah,al
		mov	al,8
		out	dx,ax
		mov	dl,0cch
		in	al,dx
		mov	dl,0c2h
		and	al,0dfh
		shr	cl,1
		jc	nob2
		or	al,20h
nob2:		out	dx,al
		sti
		pop	cx
		pop	dx
		pop	ax
		ret
everex_bank	endp

paradise_bank	proc			;Paradise
		cli
		mov	SS:svga_cur_bank,al
		push	ax
		push	dx
		mov	dx,3ceh
		mov	ax,50fh		;turn off write protect on VGA registers
		out	dx,ax
		mov	ah,SS:svga_cur_bank
		shl	ah,1
		shl	ah,1
		shl	ah,1
		shl	ah,1
		mov	al,9
		out	dx,ax
		sti
		pop	dx
		pop	ax
		ret
paradise_bank	endp

trident_bank	proc			;Trident
		cli
		mov	SS:svga_cur_bank,al
		push	ax
		push	dx
		mov	dx,3ceh		;set page size to 64k
		mov	al,6
		out	dx,al
		inc	dl
		in	al,dx
		dec	dl
		or	al,4
		mov	ah,al
		mov	al,6
		out	dx,ax
		
		mov	dl,0c4h		;switch to BPS mode
		mov	al,0bh
		out	dx,al
		inc	dl
		in	al,dx
		dec	dl

		mov	ah,SS:svga_cur_bank
		xor	ah,2
		mov	dx,3c4h
		mov	al,0eh
		out	dx,ax
		sti
		pop	dx
		pop	ax
		ret
trident_bank	endp

tseng_bank	proc			;Tseng
		cli
		mov	SS:svga_cur_bank,al
		push	ax
		push	dx
		and	al,7
		mov	ah,al
		shl	ah,1
		shl	ah,1
		shl	ah,1
		or	al,ah
		or	al,01000000b
		mov	dx,3cdh
		out	dx,al
		sti
		pop	dx
		pop	ax
		ret
tseng_bank	endp

video7_bank	proc			;Video 7
		cli
		mov	SS:svga_cur_bank,al
		push	ax
		push	dx
		push	cx
video7xx:
		and	ax,15
		mov	ch,al
		mov	dx,3c4h
		mov	ax,0ea06h
		out	dx,ax
		mov	ah,ch
		and	ah,1
		mov	al,0f9h
		out	dx,ax
		mov	al,ch
		and	al,1100b
		mov	ah,al
		shr	ah,1
		shr	ah,1
		or	ah,al
		mov	al,0f6h
		out	dx,al
		inc	dx
		in	al,dx
		dec	dx
		and	al,not 1111b
		or	ah,al
		mov	al,0f6h
		out	dx,ax
		mov	ah,ch
		mov	cl,4
		shl	ah,cl
		and	ah,100000b
		mov	dl,0cch
		in	al,dx
		mov	dl,0c2h
		and	al,not 100000b
		or	al,ah
		out	dx,al
		sti
		pop	cx
		pop	dx
		pop	ax
		ret
video7_bank	endp
	
oaktech_bank	proc			;Oak Technology Inc OTI-067
		cli
		mov	SS:svga_cur_bank,al
		push	ax
		push	dx
		and	al,15
		mov	ah,al
		shl	al,1
		shl	al,1
		shl	al,1
		shl	al,1
		or	al,ah
		mov	dx,3dfh
		out	dx,al
		sti
		pop	dx
		pop	ax
		ret
oaktech_bank	endp

tseng4_bank	proc			;Tseng 4000 series
		cli
		mov	SS:svga_cur_bank,al
		push	ax
		push	dx
		mov	ah,al
		mov	dx,3bfh			;Enable access to extended registers
		mov	al,3
		out	dx,al
		mov	dl,0d8h
		mov	al,0a0h
		out	dx,al
		and	ah,15
		mov	al,ah
		shl	al,1
		shl	al,1
		shl	al,1
		shl	al,1
		or	al,ah
		mov	dl,0cdh
		out	dx,al
		sti
		pop	dx
		pop	ax
		ret
tseng4_bank	endp



;===============================================
; Calc pixel address of pixel (x, y).
; Called internally only.
;
; Called with:
;	 		BX	- X
;			AX	- Y
; Returns:
;			ES:DI	- Video buffer address
;			DX	- Number of bytes remaining in video bank
;

svga_paddr	proc	near
		clc
		mul	SS:adapter_x         	; start line * number of pixels per line
		add	ax,bx			; add X offset
		adc	dx,0			; add carry
		mov	di,ax
		push    ax               	; save pixel address for later
		cmp     dl,SS:svga_cur_bank	; bank same as previous ?
		je      svga_paddr1		; jump if bank same as previous
		mov     al,dl            	; put bank in al
		call	SS:[adapter_bank]	; map in new bank
svga_paddr1:
		pop	ax
		mov	dx,0FFFFH		; Size of a bank - 1
		sub	dx,ax			; Find bytes remaining in bank
		inc	dx			; i.e. number of bytes left to go until we have to use next bank
		ret
svga_paddr	endp



svga_putline	proc    uses di si ds es, p:FAR PTR, n:WORD, x:WORD, y:WORD ; super VGA 256 colors
		cld
		mov	bx,x
		mov	ax,y
		call	svga_paddr		; Calc start addr of pixel
		mov	ax,0A000H		; Video buffer segment
		mov	es,ax
		lds	si,p			; Load ptr to scan line
		mov	cx,n			; Number of bytes to move
		or	dx,dx
		jz	svga_putline2		; JIF 64K bytes remain in bank 
		cmp	cx,dx			; Test # bytes to move with # available in current bank
		jbe	svga_putline2		; JIF this line will fit in current bank
		sub	cx,dx			; Calc # bytes to put into next bank
		xchg	cx,dx			; Save it in DX and set up CX for this bank
		shr	cx,1			; Convert # bytes to # words
		jnc	svga_putline1
		movsb				; Copy the first byte
svga_putline1:
		rep	movsw			; Copy the remainder of the line (best if ptr is word aligned too!)
		mov	al,SS:svga_cur_bank	; Get current bank
		inc	al
		call	SS:[adapter_bank]	; Swap to next bank
		mov	cx,dx			; Set # of bytes to copy to new bank
svga_putline2:
		shr	cx,1			; Convert # bytes to # words
		jnc	svga_putline3
		movsb				; Copy the first byte
svga_putline3:
		rep	movsw			; Copy the remainder of the line (best if ptr is word aligned too!)

		ret
svga_putline	endp


svga_getline	proc    uses di si ds es, p:FAR PTR, n:WORD, x:WORD, y:WORD ; super VGA 256 colors
		cld
		mov	bx,x
		mov	ax,y
		call	svga_paddr		; Calc start addr of pixel
		mov	ax,0A000H		; Video buffer segment
		mov	es,ax
		lds	si,p			; Load ptr to scan line
		mov	cx,n			; Number of bytes to move
		cmp	cx,dx			; Test # bytes to move with # available in current bank
		jbe	svga_getline2		; JIF this line will fit in current bank
		sub	cx,dx			; Calc # bytes to put into next bank
		xchg	cx,dx			; Save it in DX and set up CX for this bank
		shr	cx,1			; Convert # bytes to # words
		jnc	svga_getline1
		movsb				; Copy the first byte
svga_getline1:
		rep	movsw			; Copy the remainder of the line (best if ptr is word aligned too!)
		mov	al,SS:svga_cur_bank	; Get current bank
		inc	al
		call	SS:[adapter_bank]	; Swap to next bank
		mov	cx,dx			; Set # of bytes to copy to new bank
svga_getline2:
		shr	cx,1			; Convert # bytes to # words
		jnc	svga_getline3
		movsb				; Copy the first byte
svga_getline3:
		rep	movsw			; Copy the remainder of the line (best if ptr is word aligned too!)
		ret
svga_getline	endp


		SUBTTL	svga_EMSdecode
		PAGE

		COMMENT	$

Name:		svga_EMSdecode

Function:	Decodes a VORT run-length encoded colour mapped image
		from EMS memory to PVGA front or back image buffer.

Caller:		Microsoft C

		void	svga_EMSdecode(EMS_h, EMS_ofs, x, y, xlen, ylen)
		UINT	EMS_h;			/* EMS handle */
		ULONG	EMS_ofs;		/* linear offset with EMS pages */
		int	x, y, xlen, ylen;	/* position and size */

		$


svga_EMSdecode	proc	uses di si ds es, EMS_h:WORD, EMS_ofs:DWORD, x:WORD, y:WORD, xlen:WORD, ylen:WORD
		LOCAL	j:WORD
		LOCAL	i:WORD
		LOCAL	pg:WORD
		LOCAL	base_ofs:WORD
		LOCAL	len_l:WORD
		LOCAL	len_h:WORD
		LOCAL	start_bank:BYTE

		call	EMS_base	; Get address of page frame into DX:AX
		push	dx		; Save segment part of EMS base addr
		mov	si,ax		; Setup DS:SI with source data pointer
		mov	base_ofs,ax	; Save offset for later reference

					; Calculate page # and page offset
		mov	ax,WORD PTR (EMS_ofs)
		mov	dx,WORD PTR (EMS_ofs+2)
		mov	bx,ax
		and	bx,03FFFH	; Lower 14 bits are offset into page
		add	si,bx
		rcl	ax,1		; Upper 18 bits is page 
		rcl	dx,1
		rcl	ax,1
		rcl	dx,1
		mov	pg,dx
		push	dx
		mov	ax,EMS_h	; Map in starting page
		push	ax
		call	EMS_map	; Map in EMS page (we presume no errors)
		add	sp,4

		mov	ax,ylen		; Width of image
		mov	j,ax

					; calculate 1st pixel address and map in bank
		mov	bx,x
		mov	ax,y
		call	svga_paddr	; Video bank ptr is ES:DI, DX bytes remaining in the bank
		mov	ax,0A000H		; Video buffer segment
		mov	es,ax

decode_it:
		pop	ds		; Load EMS base address segment
		mov	ch,0		; Zap high byte of repeat counter

		GetEMSByte EMS_h, pg	; Load 4 byte image length value (one byte at a time)
		mov	ah,al
		GetEMSByte EMS_h, pg
		xchg	ah,al
		mov	len_l,ax	; Save LSB

		GetEMSByte EMS_h, pg
		mov	ah,al
		GetEMSByte EMS_h, pg
		xchg	ah,al
		mov	len_h,ax	; Save MSB

decode_1:
		cmp	len_l,0
		jnz	@F		; Continue if not zero
		cmp	len_h,0
		jnz	@F
		jmp	decode_x	; Exit if no more bytes in image
@@:
		mov	bx,xlen		; X width
					; Save state at start of this line
		push	dx		; Save number of bytes remaining in bank
		push	di		; Save destination offset
		mov	al,SS:svga_cur_bank	; Save video bank we are starting within
		mov	start_bank,al

decode_2:
		cmp	bx,0
		jnz	decode_2_1
		jmp	decode_10	; JIF finished this row
decode_2_1:
		; Put in a test for LEN_L == 0 ???
		sub	len_l,1		; One less byte in the image data
		jnc	@F
		dec	len_h
@@:
		GetEMSByte EMS_h, pg	; Get next byte (rnnnnnnn)
		mov	cl,al		; Run length (?nnnnnnn)
		test	al,80H		; Test top bit (r???????)
		jnz	decode_3	; JIF top bit set

				; Replicate next byte n+1 times
		inc	cx		; Run of 0 bytes is actually a run of 1 byte
					; Set next "run" bytes of image to *(DS:SI)
		sub	bx,cx		; Calc bytes remaining on this row
		sub	len_l,1		; One less byte in the image data
		jnc	@F
		dec	len_h
@@:
		GetEMSByte EMS_h, pg	; Get value to replicate

		or	dx,dx		; Test if we are the start of bank
		jz	decode_2_2	; JIF there is plenty of room
		cmp	cx,dx		; Ensure that there is enough room in the current video bank
		jb	decode_2_2	; JIF there is room
		sub	cx,dx		; Calc # bytes to shove into next bank
		push	cx
		mov	cx,dx
		rep	stosb		; Fill the last of the current bank
		push	ax
		mov	al,SS:svga_cur_bank
		inc	al
		call	SS:[adapter_bank]; Map in next bank
		pop	ax		; Restore value to replicate
		pop	cx		; Restore remainder to fill
		mov	di,0		; Start of bank
		mov	dx,0000H	; Number of bytes in new bank (64K)
		jcxz	@F		; Don't replicate if none
decode_2_2:
		sub	dx,cx		; Calc bytes remaining in new bank
		rep	stosb		; Replicate
@@:
		jmp	decode_2	; Go for next bit of encoded image


decode_3:			; Copy next n bytes from EMS to video
		and	cl,07FH		; Zap top bit - make run length
		sub	bx,cx		; Calc bytes remaining on this row
		sub	len_l,cx	; Bytes remaining in image
		jnc	@F
		dec	len_h
@@:
		push	dx		; CX is bytes to copy - ensure we don't run over end of EMS page
		mov	dx,si
		add	dx,cx		; Calc end byte of run
		test	dx,04000H	; Over the end of the EMS page?
		jnz	@F
		pop	dx		; Restore # bytes in video bank
		jmp	short decode_4	; JIF run will fit into current EMS page
@@:					; We will run over into next page
		sub	dx,4000H	; # bytes into next page
		mov	cx,4000H	; # bytes to end of current page
		sub	cx,si
		jcxz	decode_3_3

		pop	ax		; Restore # bytes in video bank
		xchg	dx,ax
		push	ax		; Save # bytes into next EMS page

		or	dx,dx		; Test if we are the start of bank
		jz	decode_3_1	; JIF there is plenty of room
		cmp	cx,dx		; Ensure that there is enough room in the current video bank
		jb	decode_3_1	; JIF there is room
		je	decode_3_2	; JIF there is room (but we end at the end of the bank)
		sub	cx,dx		; Calc # bytes to shove into next bank
		push	cx
		mov	cx,dx
		rep	movsb		; Move the last of the current bank
		mov	al,SS:svga_cur_bank
		inc	al
		call	SS:[adapter_bank]; Map in next bank
		pop	cx		; Restore remainder to move
		mov	di,0		; Start of bank
		mov	dx,0000H	; Number of bytes in new bank (64K)
decode_3_1:
		sub	dx,cx		; Calc bytes remaining in new bank
		rep	movsb		; Move
		jmp	short decode_3_3; Go and map in next EMS page

decode_3_2:
		rep	movsb		; Move
		mov	al,SS:svga_cur_bank
		inc	al
		call	SS:[adapter_bank]; Map in next bank
		mov	di,0		; Start of bank
		mov	dx,0000H	; Number of bytes in new bank (64K)

decode_3_3:
;;;;;		pop	cx		; Remainder to copy from new page
		mov	si,base_ofs	; Start at beginning of new page
;;;;;		push	cx
		push	es
		push	di
		inc	pg		; Map in next EMS page
		mov	ax,pg
		push	ax
		mov	ax,EMS_h
		push	ax
		call	EMS_map
		add	sp,4
		pop	di
		pop	es
		pop	cx
decode_4:
		jcxz	decode_4_3	; Test for safety sake

		or	dx,dx		; Test for start of bank
		jz	decode_4_1	; JIF there is plenty of room
		cmp	cx,dx		; Ensure that there is enough room in the current video bank
		jb	decode_4_1	; JIF there is room
		sub	cx,dx		; Calc # bytes to shove into next bank
		push	cx
		mov	cx,dx
		rep	movsb		; Move the last of the current bank
		mov	al,SS:svga_cur_bank
		inc	al
		call	SS:[adapter_bank]; Map in next bank
		pop	cx		; Restore remainder to move
		mov	di,0		; Start of bank
		mov	dx,0000H	; Number of bytes in new bank (64K)
		jcxz	decode_4_3	; Don't move bytes if there is none
decode_4_1:
		sub	dx,cx		; Calc bytes remaining in new bank
		rep	movsb		; Move

decode_4_3:
		jmp	decode_2	; Go for next bit of encoded image


decode_10:				; End of image line
		dec	j		; One less row
		; Test for j == 0 here ?

					; Restore state at start of the line just output
		pop	di		; Restore buffer address of start of destination line
		pop	dx		; And restore # bytes remaining in the bank then too
		sub	dx,SS:adapter_x	; Bytes now remaining in bank
		add	di,SS:adapter_x	; Next video buffer line
		jc	decode_11	; JIF starts in next bank
		jmp	decode_1	; Go for next line
decode_11:
		mov	al,start_bank	; Swap in next bank
		inc	al
		call	SS:[adapter_bank]
		jmp	decode_1	; Go for next line

decode_x:
		ret

svga_EMSdecode	endp


; **************** Video adapter detection routine ***********************
;
;	This routine performs a few quick checks on the type of
;	video adapter installed.  It returns AX with a value indicating
;	the adapter found.
;


adapter_detect	proc	uses si es ds
		mov	ax,1a00h		; look for a VGA/MCGA
		int	10h			;  by using a VGA-specific call
		cmp	al,1ah			; was AL modified?
		je	adapter_detect_1	;  Yup.  A VGA or an MCGA
		mov	ax,ADAPTER_NONE		; Not a video card we care to know about
		ret

adapter_detect_1:			; Determine type of VGA
		mov	ax,0c000h		; Test for ATI
		mov	es,ax
		cmp	word ptr es:[40h],'13'
		jnz	noati
		mov	ax,ADAPTER_ATI
		jmp	fini

noati:		mov	ax,7000h		;Test for Everex
		xor	bx,bx
		cld
		int	10h
		cmp	al,70h
		jnz	noev
		mov	ax, ADAPTER_EVEREX
						; Test for Trident
		cmp	dx,6780h
		jz	yeste
		cmp	dx,2360h
		jnz	note
yeste:
		mov	ax,ADAPTER_TRIDENT
note:		jmp	fini

noev:		mov	dx,3c4h			;Test for Trident
		mov	al,0bh
		out	dx,al
		inc	dl
		in	al,dx
		cmp	al,0fh
		ja	notri
		cmp	al,2
		jb	notri
		mov	ax,ADAPTER_TRIDENT
		jmp	fini

notri:		mov	ax,6f00h		;Test for Video 7
		xor	bx,bx
		cld
		int	10h
		cmp	bx,'V7'
		jnz	nov7
		mov	ax,ADAPTER_VIDEO7
		jmp	fini

nov7:		mov	dx,3ceh			;Test for Paradise
		mov	al,9			;check Bank switch register
		out	dx,al
		inc	dx
		in	al,dx
		dec	dx
		or	al,al
		jnz	nopd

		mov	ax,50fh			;turn off write protect on VGA registers
		out	dx,ax
						; Check Paradise banks
						;if bank 0 and 1 same not paradise
		mov	cx,1
		mov	di,0b800h
		mov	es,di
		xor	di,di
		mov	bx,1234h
		call	$gochk
		jnz	nopd
		mov	bx,4321h
		call	$gochk
		jnz	nopd
		mov	ax,ADAPTER_PARADISE
		jmp	fini

nopd:		mov	ax,5f00h		;Test for Chips & Tech
		xor	bx,bx
		cld
		int	10h
		cmp	al,5fh
		jnz	noct
		mov	ax,ADAPTER_CHIPS
		jmp	fini

noct:		mov	ch,0
		mov	dx,3d4h			;check for Tseng 4000 series
		mov	al,33h
		out	dx,al
		inc	dx
		in	al,dx
		dec	dx
		mov	cl,al
		mov	ax,00a33h
		out	dx,ax
		mov	al,33h
		out	dx,al
		inc	dx
		in	al,dx
		and 	al,0fh
		dec	dx
		cmp	al,00ah
		jnz	noct2
		mov	ax,00533h
		out	dx,ax
		mov	al,33h
		out	dx,al
		inc	dx
		in	al,dx
		dec	dx
		and 	al,0fh
		cmp	al,005h
		jnz	noct2
		mov	al,33h
		mov	ah,cl
		out	dx,ax
		mov	ch,1

		mov	dx,3bfh			;Enable access to extended registers
		mov	al,3
		out	dx,al
		mov	dl,0d8h
		mov	al,0a0h
		out	dx,al

noct2:		mov	dx,3cdh			;Test for Tseng
		in	al,dx			;save bank switch register
		mov	cl,al
		mov	al,0aah			;test register with 0aah
		out	dx,al
		in	al,dx
		cmp	al,0aah
		jnz	nots
		mov	al,055h			;test register with 055h
		out	dx,al
		in	al,dx
		cmp	al,055h
		jnz	nots
		mov	al,cl
		out	dx,al
		mov	ax,ADAPTER_TSENG

		cmp	ch,0
		jz	oldts
		mov	ax,ADAPTER_TSENG4
oldts:		jmp	fini

nots:	
        	mov     dx,3ceh		;Test for Above A or B chipsets
        	mov     ax,200fh
        	out     dx,ax
        	inc	dx
		nojmp
        	in      al,dx
		cmp	al,21h
		jz	verb
		cmp	al,20h
		jnz	noab
		mov	ax,ADAPTER_AHEADA
		jmp	short fini

verb:		mov	ax,ADAPTER_AHEADB
		jmp	short fini

noab:		mov	dx,3deh
		in	al,dx
		and	al,11100000b
		cmp	al,01100000b
		jnz	nooak
		mov	ax,ADAPTER_OTI
		jmp	short fini
nooak:
		mov	ax,ADAPTER_VGA		; Plain vanilla VGA
fini:
		ret
adapter_detect	endp



;----------------------------------------
; Paradise bank checking helper routines.
;
$gochk		proc	near
		push	si
		mov	si,bx

		mov	al,cl
		call	$pdrsub
		xchg	bl,es:[di]
		mov	al,ch
		call	$pdrsub
		xchg	bh,es:[di]

		xchg	si,bx

		mov	al,cl
		call	$pdrsub
		xor	bl,es:[di]
		mov	al,ch
		call	$pdrsub
		xor	bh,es:[di]

		xchg	si,bx

		mov	al,ch
		call	$pdrsub
		mov	es:[di],bh
		mov	al,cl
		call	$pdrsub
		mov	es:[di],bl

		mov	al,0
		call	$pdrsub
		or	si,si
		pop	si
		ret
$gochk		endp


$pdrsub		proc	near		;Paradise
		mov	ah,al
		mov	dx,3ceh
		mov	al,9
		out	dx,ax
		ret
$pdrsub		endp


 		end

