/* simple program to animate a series of frames on an Iris
  by Pavel Rozalski
  pavelr@sco.com

  This should be compiled with -DZBUFFER in general
*/

#include <stdio.h>
#include <gl.h>
#include <device.h>
#include "vort.h"

typedef struct _screenPos {
	Screencoord x1, y1;
} screenPos;

screenPos *srcImages;

int nx, ny, w, h, magx=1, magy=1, maxframes, numframes;
char *r, *g, *b;

#ifdef ZBUFFER
int lastz;
#endif /* ZBUFFER */

extern char *malloc ();

Usage ()
{
	fprintf (stderr, "movie [-x magx] [-y magy] [-m mag] pic1 pic2 ...\n");
	exit ();
}

main (argc, argv)
int argc;
char **argv;
{
	int i;
	extern char *optarg;
	extern int optind;
	int c;

	while ((c = getopt (argc, argv, "x:y:m:")) != -1)
		switch (c) {
		case 'x':
			magx = atoi (optarg);
			break;
	
		case 'y':
			magy = atoi (optarg);
			break;
	
		case 'm':
			magx = magy = atoi (optarg);
			break;
		case '?':
			Usage ();
			break;
		}
	
	if ((magx<=0) || (magy<=0)) {
		fprintf (stderr, "Magnification must be positive\n");
		exit ();
	}

	InitTable ((char *) argv[optind]);
	
	InitGraphics ();

	numframes = 0;

	for (i=optind; i < argc && numframes<maxframes; i++)
		LoadImage ((char *) argv[i]);

	rectzoom ((float) magx, (float) magy);

#ifdef	ZBUFFER
	DisplayZ ();
#else
	Display ();
#endif	/* ZBUFFER */
}

#ifdef	ZBUFFER
DisplayZ ()
{
	int i;

	qdevice (ESCKEY);

	zdraw (FALSE);
	frontbuffer (TRUE);

	for (i=0; ;){
		if (i==0) {
			readsource (SRC_ZBUFFER);
		} else if (i==lastz) {
			readsource (SRC_FRONT);
		}

		gsync ();
		rectcopy (srcImages[i].x1, srcImages[i].y1,
			  srcImages[i].x1+w-1, srcImages[i].y1+h-1,
			  0, 0);
		gsync ();
		i++;
		if (i==numframes) i=0;
		if (qtest())
			exit ();
	}
}
#else	/* !ZBUFFER */
Display ()
{
	int i;

	for (i=0; ;){
		gsync ();
		rectcopy (srcImages[i].x1, srcImages[i].y1,
			  srcImages[i].x1+w-1, srcImages[i].y1+h-1,
			  0, 0);
		gsync ();
		i++;
		if (i==numframes) i=0;
	}
}
#endif	/* ZBUFFER */

/* find the x and y size of the image. Compute some tables */
InitTable (fname)
char *fname;
{
	int x, y;
	image *im;

	if ((im = openimage (fname, "r")) == (image *) NULL) {
		fprintf (stderr, "Unable to open image %s\n", fname);
		exit ();
	}

        w = imagewidth(im);
        h = imageheight(im);

	nx = (XMAXSCREEN+1) / w;
	ny = (YMAXSCREEN+1) / h;


	if ((magx>nx) || (magy>ny)) {
		fprintf (stderr, "Magnification too large\n");
		exit ();
	}

	closeimage (im);

	r = (char *) malloc (w);
	g = (char *) malloc (w);
	b = (char *) malloc (w);

#ifdef	ZBUFFER
	maxframes = 2 * nx * ny - magx * magy;
#else
	maxframes =     nx * ny - magx * magy;
#endif	/* ZBUFFER */

	numframes = 0;

	srcImages = (screenPos *) malloc (maxframes * sizeof (screenPos));

#ifdef	ZBUFFER
	/* calculate all frame positions in Zbuffer - all of it is available */
	for (y=0; y <ny; y++)
		for (x = 0; x<nx; x++) {
			srcImages[numframes].x1 = x * w;
			srcImages[numframes].y1 = y * h;
			numframes ++;
		}
	lastz = numframes;
#endif	/* ZBUFFER */

	/* we have to skip over some regions where the target image will be */
	for (y= 0; y<magy ; y++)
		for (x= magx; x<nx; x++) {
			srcImages[numframes].x1 = x * w;
			srcImages[numframes].y1 = y * h;
			numframes ++;
		}

	/* now do the rest of the frame buffer */
	for (y=magy; y<ny; y++)
		for (x = 0; x<nx; x++) {
			srcImages[numframes].x1 = x * w;
			srcImages[numframes].y1 = y * h;
			numframes ++;
		}
}

InitGraphics ()
{
	ginit ();
	RGBmode ();
	gconfig ();


	RGBcolor (0, 0, 0);

#ifdef	ZBUFFER
	czclear ();
	zdraw (TRUE);
#else	/* ZBUFFER */
	clear ();
#endif	/* ZBUFFER */
}

LoadImage (fname)
char *fname;
{
	image *im;
	int y;
	if ((im = openimage (fname, "r")) == (image *) NULL) {
		fprintf (stderr, "Unable to open image %s\n", fname);
		return;
	}

	if ((imagewidth(im) != w) || (imageheight(im) != h)) {
		fprintf (stderr, "Image wrong size: %s\n", fname);
		closeimage (im);
		return;
	}

	for (y=h; y>=0; y--) {
		readrgbline (im, r, g, b);
		cmov2i (srcImages[numframes].x1, srcImages[numframes].y1+y);
		writeRGB ((short) w, r, g, b);
	}
	closeimage (im);
	numframes ++;

#ifdef	ZBUFFER
	if (numframes == lastz) {
		zdraw (FALSE);
		frontbuffer (TRUE);
	}
#endif	/* ZBUFFER */
}
