/******************************************************************************
* File: triggers.c                                                            *
* All functionality for the trigger system.                                   *
*                                                                             *
* This file is Copyright (C) 1997 by Joachim Pileborg, all rights reserved.   *
******************************************************************************/

#define __TRIGGERS_C__

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "utils.h"
#include "triggers.h"

/*****************************************************************************/

/* external variables */
extern struct char_data *character_list;
extern struct obj_data  *object_list;
extern struct room_data *world;
extern int top_of_world;

/* external functions */
int execute_script(const char *name, const char *func, int source, void *me,
									 int room, struct obj_data *obj, struct char_data *ch,
									 char *txt);

/*****************************************************************************/

const char *trigger_types[] =
{
	"TIME",
	"ENTER",
	"LEAVE",
	"TELL",
	"ASK",
	"SAY",
	"GIVE",
	"\n"
};

/*****************************************************************************/

/* functions private to this file */
static void _trigger_time(struct trigger_data *t_list);
static int _trigger_enter_leave(struct char_data *who,
																struct trigger_data *l, int type);

/* variables private to this file */

/*****************************************************************************/

/*
 * FUNCTION:    trigger_time()
 * DESCRIPTION: Walk through the list of all triggers, and execute all
 *              time triggers.  This function is 
 */
void trigger_time(void)
{
	register struct char_data *ch, *next_ch;
	register struct obj_data  *ob, *next_ob;
	register int               room;

	/* go through the character list */
	for (ch = character_list; ch; ch = next_ch)
	{
		next_ch = ch->next;

		if (IS_NPC(ch) && MOB_TRIGGERS(ch))
		{
			_trigger_time(MOB_TRIGGERS(ch));
		}
	}

	/* go through the object list */
	for (ob = object_list; ob; ob = next_ob)
	{
		next_ob = ob->next;

		if (OBJ_TRIGGERS(ob))
		{
			_trigger_time(OBJ_TRIGGERS(ob));
		}
	}

	/* go through the world */
	for (room = 0; room <= top_of_world; room++)
	{
		if (ROOM_TRIGGERS(room))
		{
			_trigger_time(ROOM_TRIGGERS(room));
		}
	}
}

/*****************************************************************************/

int trigger_enter_leave(struct char_data *who, int in_room, int type)
{
	register struct char_data *ch, *next_ch;
	register struct obj_data  *ob, *next_ob;

	/* check the room */
	if (ROOM_TRIGGERS(in_room))
	{
		if (!_trigger_enter_leave(who, ROOM_TRIGGERS(in_room), type))
		{
			return 0;
		}
	}

	/* go through the character list */
	for (ch = world[in_room].people; ch; ch = next_ch)
	{
		next_ch = ch->next_in_room;

		if (IS_NPC(ch) && MOB_TRIGGERS(ch))
		{
			if (!_trigger_enter_leave(who, MOB_TRIGGERS(ch), type))
			{
				return 0;
			}
		}
	}

	/* go through the object list */
	for (ob = world[in_room].contents; ob; ob = next_ob)
	{
		next_ob = ob->next_content;

		if (OBJ_TRIGGERS(ob))
		{
			if (!_trigger_enter_leave(who, OBJ_TRIGGERS(ob), type))
			{
				return 0;
			}
		}
	}

	return 1;
}


/*****************************************************************************/

/*****************************************************************************/

struct trigger_data *trigger_create(int type, char *script, void *ob,
																		int source)
{
	struct trigger_data *t;
	char *ptr;

	CREATE(t, struct trigger_data, 1);
	t->type   = type;
	t->me     = ob;
	t->source = source;

	if ((ptr = strchr(script, ':')) != NULL)
	{
		*ptr = 0;
		t->script = strdup(script);
		t->func   = strdup(ptr + 1);
	}
	else
	{
		t->script = strdup(script);
	}

	return t;
}

/*****************************************************************************/

void trigger_add(struct trigger_data *t, struct trigger_data **l)
{
	struct trigger_data *trigger;

	t->next = *l;
	*l = t;

	for (trigger = *l; trigger; trigger = trigger->next)
	{
		trigger->list = t;
	}
}

/*****************************************************************************/


void trigger_remove(register struct trigger_data *t)
{
	register struct trigger_data *temp;

	REMOVE_FROM_LIST(t, t->list, next);

	if (t->script)
	{
		free(t->script);
	}
	if (t->func)
	{
		free(t->func);
	}

	free(t);
}

/*****************************************************************************/

static void _trigger_time(struct trigger_data *t_list)
{
	register struct trigger_data *t;
	register struct trigger_data *next_t;

	for (t = t_list; t; t = next_t)
	{
		next_t = t->next;

		if (t->type == TT_TIME)
		{
			if (execute_script(t->script, t->func, t->source, t->me,
												 NOWHERE, NULL, NULL, NULL) < 0)
			{
				/* couldn't execute the script, remove the trigger */
				trigger_remove(t);
			}
		}
	}
}

/*****************************************************************************/

static int _trigger_enter_leave(struct char_data *who,
																struct trigger_data *l, int type)
{
	register struct trigger_data *t      = NULL;
	register struct trigger_data *next_t = NULL;
	int res = 0;

	for (t = l; t; t = next_t)
	{
		next_t = t->next;

		if (t->type == type)
		{
			if ((res = execute_script(t->script, t->func, t->source, t->me,
												 NOWHERE, NULL, who, NULL)) < 0)
			{
				/* couldn't execute the script, remove the trigger */
				trigger_remove(t);
			}
			else if (res == 0)
			{
				return 0;
			}
		}
	}

	return 1;
}

/*****************************************************************************/
