/*
 * Copyright (C) 1998-2002 Luke Howard.
 * This file is part of the pam_ldap library.
 * Contributed by Luke Howard, <lukeh@padl.com>, 1998.
 *
 * The pam_ldap library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The pam_ldap library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with the pam_ldap library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _LDAPFCN_H_
#define _LDAPFCN_H_

#ifdef USE_LDAP

/* /etc/ldap.conf nss_ldap-style configuration */
typedef struct ldap_config
  {
    /* URI */
    char *uri;
    /* space delimited list of servers */
    char *host;
    /* port, expected to be common to all servers */
    int port;
    /* base DN, eg. dc=gnu,dc=org */
    char *base;
    /* scope for searches */
    int scope;
    /* deref policy */
    int deref;
    /* bind dn/pw for "anonymous" authentication */
    char *binddn;
    char *bindpw;
    /* bind dn/pw for "root" authentication */
    char *rootbinddn;
    char *rootbindpw;
    /* SSL config states */
#define SSL_OFF          0
#define SSL_LDAPS        1
#define SSL_START_TLS    2
    int ssl_on;
    /* SSL path */
    char *sslpath;
    /* filter to AND with uid=%s */
    char *filter;
    /* attribute to search on; defaults to uid. Use CN with ADS? */
    char *userattr;
    /* attribute to set PAM_USER based on */
    char *tmplattr;
    /* default template user */
    char *tmpluser;
    /* search for Netscape password policy */
    int getpolicy;
    /* host attribute checking, for access authorization */
    int checkhostattr;
    /* group name; optional, for access authorization */
    char *groupdn;
    /* group membership attribute; defaults to uniquemember */
    char *groupattr;
    /* LDAP protocol version */
    int version;
    /* search timelimit */
    int timelimit;
    /* bind timelimit */
    int bind_timelimit;
    /* automatically chase referrals */
    int referrals;
    /* restart interrupted syscalls, OpenLDAP only */
    int restart;
    /* chauthtok config states */
#define PASSWORD_CLEAR   0
#define PASSWORD_CRYPT   1
#define PASSWORD_MD5     2
#define PASSWORD_NDS     3
#define PASSWORD_AD      4
#define PASSWORD_EXOP    5
    int password_type;
    /* min uid */
    uid_t min_uid;
    /* max uid */
    uid_t max_uid;
    /* tls check peer */
    int tls_checkpeer;
    /* tls ca certificate file */
    char *tls_cacertfile;
    /* tls ca certificate dir */
    char *tls_cacertdir;
    /* tls ciphersuite */
    char *tls_ciphers;
    /* tls certificate */
    char *tls_cert;
    /* tls key */
    char *tls_key;
  }
ldap_config_t;

/* Netscape global password policy attributes */
typedef struct ldap_password_policy
  {
    int password_change;
    int password_check_syntax;
    int password_min_length;
    int password_exp;
    int password_max_age;
    int password_warning;
    int password_keep_history;
    int password_in_history;
    int password_lockout;
    int password_max_failure;
    int password_unlock;
    int password_lockout_duration;
    int password_reset_duration;
  }
ldap_password_policy_t;

/* Standard Unix style shadow controls */
typedef struct ldap_shadow
  {
    int shadowacct;		/* is shadowAccount */
    long int lstchg;		/* Date of last change.  */
    long int min;		/* Minimum number of days between changes.  */
    long int max;		/* Maximum number of days between changes.  */
    long int warn;		/* Number of days to warn user to change
				   the password.  */
    long int inact;		/* Number of days the account may be
				   inactive.  */
    long int expire;		/* Number of days since 1970-01-01 until
				   account expires.  */
    unsigned long int flag;	/* Reserved.  */
  }
ldap_shadow_t;

/* Password controls sent to client */
#ifndef LDAP_CONTROL_PWEXPIRED
#define LDAP_CONTROL_PWEXPIRED      "2.16.840.1.113730.3.4.4"
#endif /* LDAP_CONTROL_PWEXPIRED */
#ifndef LDAP_CONTROL_PWEXPIRING
#define LDAP_CONTROL_PWEXPIRING     "2.16.840.1.113730.3.4.5"
#endif /* LDAP_CONTROL_PWEXPIRING */

#ifndef LDAP_OPT_ON
#define LDAP_OPT_ON ((void *) 1)
#endif /* LDPA_OPT_ON */
#ifndef LDAP_OPT_OFF
#define LDAP_OPT_OFF ((void *) 0)
#endif /* LDAP_OPT_OFF */

/* Seconds in a day */
#define SECSPERDAY 86400

/* Netscape per-use password attributes. Unused except for DN. */
typedef struct ldap_user_info
  {
    /* user name, to validate info cache */
    char *username;
    /* DN of user in directory */
    char *userdn;
    /* temporary cache of user's bind credentials for rebind function */
    char *userpw;
    /* host attribute from account objectclass */
    char **hosts_allow;
    /* seconds until password expires */
    long password_expiration_time;
    /* password expires now */
    int password_expired;
    /* bound as user DN */
    int bound_as_user;
    /* user ID */
    uid_t uid;
    /* mapped user */
    char *tmpluser;
    /* shadow stuff */
    ldap_shadow_t shadow;
  }
ldap_user_info_t;

/*
 * Per PAM-call LDAP session. We keep the user info and
 * LDAP handle cached to minimize binds and searches to
 * the directory, particularly as you can't rebind within
 * a V2 session.
 */
#if defined(_INCLUDED_FROM_LDAPCFN_C_)
typedef struct ldap_session
  {
    LDAP *ld;
    ldap_config_t *conf;
    ldap_user_info_t *info;
  }
ldap_session_t;
#else
typedef struct ldap_session_t ldap_session_t;
#endif

ldap_config_t *read_ldap_config (const char *);
void free_ldap_config (ldap_config_t **pconfig);
ldap_session_t *create_ldap_session (const char *configFile);
int open_ldap_session (ldap_session_t *);
#ifndef HAVE_LDAP_GET_LDERRNO
int ldap_get_lderrno (LDAP * ld, char **m, char **s);
#endif
int get_ldapuser_info (ldap_session_t *, const char *);
int ldap_authentication (ldap_session_t *, const char *, const char *);
int ldap_update_shell (ldap_session_t *session, const char *user,
		       const char *old_password, const char *new_shell);
int ldap_update_gecos (ldap_session_t *session, const char *user,
		       const char *old_password, const char *new_gecos);
int ldap_update_field (ldap_session_t *sessoin, const char *user,
		       const char *binddn, const char *bindpw,
		       const char *field, const char *new_value);


#ifndef _pam_overwrite
#define _pam_overwrite(x) \
{ \
     register char *__xx__; \
     if ((__xx__=x)) \
          while (*__xx__) \
               *__xx__++ = '\0'; \
}
#endif

#ifndef _pam_drop
#define _pam_drop(X) \
if (X) { \
    free(X); \
    X=NULL; \
}
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifndef TRUE
#define TRUE !FALSE
#endif

#endif /* USE_LDAP */
#endif /* _LDAPFCN_H_ */
