#ifdef RCSID
static char RCSid[] =
"$Header: d:/cvsroot/tads/TADS2/OSGEN.C,v 1.3 1999/07/11 00:46:30 MJRoberts Exp $";
#endif

/* Copyright (c) 1990 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  osgen  - Operating System dependent functions, general implementation
Function
  This module contains certain OS-dependent functions that are common
  between several systems.  Routines in this file are selectively enabled
  according to macros defined in os.h:

    USE_STDIO     - implement os_print, os_flush, os_gets with stdio functions
    USE_DOSEXT    - implement os_remext, os_defext using MSDOS-like filename
                    conventions
    USE_NULLINIT  - implement os_init and os_term as do-nothing routines
    USE_NULLPAUSE - implement os_expause as a do-nothing routine
    USE_EXPAUSE   - use an os_expause that prints a 'strike any key' message
                    and calls os_waitc
    USE_TIMERAND  - implement os_rand using localtime() as a seed
    USE_NULLSTAT  - use a do-nothing os_status function
    USE_NULLSCORE - use a do-nothing os_score function
    RUNTIME       - enable character-mode console implementation  
    USE_STATLINE  - implement os_status and os_score using character-mode
                    status line implementation
    USE_OVWCHK    - implements default saved file overwrite check
    USE_NULLSTYPE - use a dummy os_settype routine
    USE_NULL_SET_TITLE - use an empty os_set_title() implementation

    If USE_STDIO is defined, we'll implicitly define USE_STDIO_INPDLG.

    If USE_STATLINE is defined, certain subroutines must be provided for
    your platform that handle the character-mode console:
        ossclr - clears a portion of the screen
        ossdsp - displays text in a given color at a given location
        ossscr - scroll down (i.e., moves a block of screen up)
        ossscu - scroll up (i.e., moves a block of screen down)
        ossloc - locate cursor

    If USE_STATLINE is defined, certain sub-options can be enabled:
        USE_SCROLLBACK - include output buffer capture in console system
        USE_HISTORY    - include command editing and history in console system
        USE_LDESC      - include long room description in status region
Notes

Modified
  01/01/98 MJRoberts     - moved certain osgen.c routines to osnoui.c  
  04/24/93 JEras         - add os_locate() for locating tads-related files
  04/12/92 MJRoberts     - add os_strsc (string score) function
  03/26/92 MJRoberts     - add os_setcolor function
  09/26/91 MJRoberts     - os/2 user exit support
  09/04/91 MJRoberts     - stop reading resources if we find '$eof' resource
  08/28/91 MJRoberts     - debugger bug fix
  08/01/91 MJRoberts     - make runstat work correctly
  07/30/91 MJRoberts     - add debug active/inactive visual cue
  05/23/91 MJRoberts     - add user exit reader
  04/08/91 MJRoberts     - add full-screen debugging support
  03/10/91 MJRoberts     - integrate John's qa-scripter mods
  11/27/90 MJRoberts     - use time() not localtime() in os_rand; cast time_t
  11/15/90 MJRoberts     - created (split off from os.c)
*/

#define OSGEN_INIT
# include "os.h"
#undef OSGEN_INIT

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>

#include "run.h"

#if defined(TURBO) || defined(DJGPP)
#include "io.h"
#endif    

#include "lib.h"
#include "tio.h"

#ifdef RUNTIME
# ifdef USE_SCROLLBACK
int osssbmode();
void scrpgup();
void scrpgdn();
void scrlnup();
void scrlndn();
static void ossdosb();
int os_f_plain = 0;
# endif /* USE_SCROLLBACK */
#endif /* RUNTIME */

/* forward declare the low-level display routine */
void ossdspn(int y, int x, int color, char *p);

/* define some dummy routines if not using graphics */
#ifndef USE_GRAPH
void os_mouhide(void) {}
void os_moushow(void) {}
#endif /* !USE_GRAPH */


/*
 *   If this port is to use the default saved file overwrite check, define
 *   USE_OVWCHK.  This routine tries to open the file; if successful, the
 *   file is closed and we ask the user if they're sure they want to overwrite
 *   the file.
 */
#ifdef USE_OVWCHK
int os_chkovw(char *filename)
{
    FILE *fp;
    
    if ((fp = fopen( filename, "r" )) != 0)
    {
        char buf[128];
        
        fclose(fp);
        os_printf("That file already exists.  Overwrite it? (y/n) >");
        os_gets((uchar *)buf, sizeof(buf));
        if (buf[0] != 'y' && buf[0] != 'Y')
            return 1;
    }
    return 0;
}
#endif /* USE_OVWCHK */

/******************************************************************************
* Ports can implement os_printf, os_flush, and os_gets as calls to the stdio
* routines of the same name by defining USE_STDIO.  These definitions can be
* used for any port for which the standard C run-time library is available.
******************************************************************************/

#ifdef USE_STDIO

/* os_printf is generally just printf(), but we broke it out to make things
*  easier for weird ports.
*/
void os_printf(const char *f, ...)
{
    va_list argptr;

    va_start(argptr, f);
    os_vprintf(f, argptr);
    va_end(argptr);
}

void os_vprintf(const char *f, va_list arglist)
{
    vprintf(f, arglist);
}

/*
 *   os_flush forces output of anything buffered for standard output.  It
 *   is generally used prior to waiting for a key (so the normal flushing
 *   may not occur, as it does when asking for a line of input).  
 */
void os_flush(void)
{
    fflush( stdout );
}

/*
 *   os_gets performs the same function as gets().  It should get a
 *   string from the keyboard, echoing it and allowing any editing
 *   appropriate to the system, and return the null-terminated string as
 *   the function's value.  The closing newline should NOT be included in
 *   the string.  
 */
uchar *os_gets(uchar *s, size_t bufl)
{
    return((uchar *)fgets((char *)s, bufl, stdin));
}

/*
 *   Get an event - stdio version.  This version does not accept a timeout
 *   value, and can only get a keystroke.  
 */
int os_get_event(unsigned long timeout, int use_timeout,
                 os_event_info_t *info)
{
    /* if there's a timeout, return an error indicating we don't allow it */
    if (use_timeout)
        return OS_EVT_NOTIMEOUT;

    /* get a key the normal way */
    info->key[0] = os_getc();

    /* if it's an extended key, get the other key */
    if (info->key[0] == 0)
        info->key[1] = os_getc();

    /* return the keyboard event */
    return OS_EVT_KEY;
}

#endif /* USE_STDIO */

/*---------------------------------------------------------------------------*/
/*
 *   Door implementation.  Allows the run-time to be used as a door from
 *   a BBS.  
 */
#ifdef USE_DOOR

int os_comport;       /* -1 indicates CON, 0 indicates COM1, 1 is COM2, etc */

static int getoutstat(void)
{
    unsigned char port_stat;
    unsigned char modem_stat;
    int           port = os_comport;
    
    if (os_comport == -1) return(0xff);
    
    asm {
        mov ah, 3
        mov dx, port
        push bp
        push es
        push ds
        push si
        push di
        int  14h
        pop  di
        pop  si
        pop  ds
        pop  es
        pop  bp
        mov port_stat, ah
        mov modem_stat, al
    }
    if (!(modem_stat & 0x80))
    {
        /* dropped carrier - abort */
        if (os_comport != -1) com_close();
        exit(0);
    }
    return(modem_stat & (1 << 4));                  /* return clear-to-send */
}

void os_printf(char *f, ...)
{
    va_list argptr;
    char    buf[256];
    char   *p;

    va_start(argptr, f);
    vsprintf(buf, f, argptr);
    va_end(argptr);

    fputs(buf, stdout);
    if (os_comport != -1)
    {
        for (p = buf ; *p ; ++p)
        {
            while (!getoutstat());
            com_out(*p);
            if (*p == '\n')
            {
                while (!getoutstat());
                com_out('\r');
            }
        }
    }
}

void os_flush(void)
{
    if (os_comport == -1) fflush(stdout);
}

#include <time.h>

char os_getc(void)
{
    time_t last_key_time = time(NULL);
    char   c;

    for (;;)
    {
        if (time(NULL) - last_key_time > 300)
        {
            os_printf("\n\n\007\007You have been idle too long.  Exiting.\n");
            if (os_comport != -1) com_close();
            exit(0);
        }
        
        if (os_comport == -1)
            c = getch();
        else
        {
            getoutstat();                      /* check for dropped carrier */
            if (!com_ready()) continue;
            c = com_in();
        }

        return(c);
    }
}

void os_waitc(void)
{
    os_getc();
}

char *os_gets(char *s, size_t bufl)
{
    unsigned char *p;
    unsigned char  c;
    long           t;
    
    for (p = s ;; )
    {
        c = os_getc();
        switch(c)
        {
        case '\r':
            *p = '\0';
            os_printf("\n");
            return(s);

        case 8:                                           /* backspace (^H) */
        case 127:                                                 /* delete */
            if (p > s)
            {
                --p;
                os_printf("\010 \010");
            }
            break;

        case 21:                                               /* control U */
            while (p > s)
            {
                --p;
                os_printf("\010 \010");
            }
            break;

        default:
            if (c >= ' ' && p < s+127)
            {
                *p++ = c;
                os_printf("%c", c);
            }
            break;
        }
    }
}

/*
 *   To prevent security problems, we implement our own directory
 *   management system.  First, the BBS must pass the user name in the
 *   environment variable TADSUSER; the format of the TADSUSER value is
 *   arbitrary -- spaces and other punctuation characters are OK.  Second,
 *   we maintain a directory file.  The format of each line in this file
 *   is simple: Username:num:apparent-name.  The num is a three-digit
 *   number which gives the actual DOS filename (of the format
 *   TADS_xxx.SAV).  The apparent-name is an arbitrary string typed by the
 *   user to identify the file; it can contain pretty much any characters
 *   other than CR.  
 */
int os_askfile(const char *prompt, char *reply, int replen,
               int prompt_type, os_filetype_t file_type)
{
    char    buf[128];
    char    buf2[128];
    char   *getenv();
    char   *uname;
    size_t  ulen;
    FILE   *fp;
    int     retval;
    long    pos;
    char   *p;
    int     id;
    int     l;

    uname = getenv("TADSUSER");
    if (!uname)
    {
        os_printf("Error:  No user information available to TADS.\n\
Please inform your SysOp.\n");
        return OS_AFE_FAILURE;
    }
    ulen = strlen(uname);

    fp = fopen("TADSFILE.DIR", "r+");
    if (!fp)
    {
        fp = fopen("TADSFILE.DIR", "w+");
        if (!fp)
        {
            os_printf("Error:  Unable to create TADS directory file.\n\
Please inform your SysOp.\n");
            return OS_AFE_FAILURE;
        }
    }

    for (;;)
    {
        os_printf("Type /L to list files, /D to delete a file\n");
        os_printf("%s > ", prompt);
        os_gets(buf, sizeof(buf));

        if (!stricmp(buf, "/L"))
        {
            fseek(fp, 0L, SEEK_SET);
            os_printf("\nListing files for user %s\n", uname);
            for (;;)
            {
                if (!fgets(buf, sizeof(buf), fp)) break;
                if ((l = strlen(buf)) && buf[l-1] == '\n') buf[l-1] = '\0';

                if (strlen(buf) > ulen && !memcmp(buf, uname, ulen)
                    && buf[ulen] == ':'
                    && memcmp(buf + ulen + 1, "xxx", 3) != 0)
                {
                    FILE *fp2;

                    sprintf(buf2, "TADS_%03d.SAV", atoi(buf + ulen + 1));
                    fp2 = fopen(buf2, "r");
                    if (fp2)
                    {
                        os_printf("   %s\n", buf + ulen + 5);
                        fclose(fp2);
                    }
                }
            }
            os_printf("\n");
        }
        else if (!stricmp(buf, "/D"))
        {
            int found;
            
            os_printf("Enter name of file to delete: ");
            os_gets(buf2, sizeof(buf2));
            buf[40] = '\0';
            if (!buf[0]) continue;
            
            fseek(fp, 0L, SEEK_SET);
            for (found = 0 ;;)
            {
                pos = ftell(fp);
                if (!fgets(buf, sizeof(buf), fp)) break;
                if ((l = strlen(buf)) && buf[l-1] == '\n') buf[l-1] = '\0';

                if (strlen(buf) > ulen && !memcmp(buf, uname, ulen)
                    && buf[ulen] == ':'
                    && memcmp(buf + ulen + 1, "xxx", 3) != 0
                    && !strcmp(buf + ulen + 5, buf2))
                {
                    fseek(fp, pos, SEEK_SET);
                    memcpy(buf + ulen + 1, "xxx", (size_t)3);
                    fputs(buf, fp);
                    os_printf("File deleted.\n");
                    found = 1;
                    break;
                }
            }
            if (!found) os_printf("File not found - nothing deleted.\n");
        }
        else if (buf[0] == '\0')
        {
            retval = OS_AFE_CANCEL;
            break;
        }
        else
        {
            int found;

            buf[40] = '\0';
            fseek(fp, 0L, SEEK_SET);
            for (found = 0, id = 0 ;;)
            {
                if (!fgets(buf2, sizeof(buf2), fp)) break;
                if ((l = strlen(buf2)) && buf2[l-1] == '\n') buf2[l-1] = '\0';

                p = strchr(buf2, ':');
                if (p && atoi(p+1) > id) id = atoi(p+1);
                
                if (strlen(buf2) > ulen && !memcmp(buf2, uname, ulen)
                    && buf2[ulen] == ':'
                    && memcmp(buf2 + ulen + 1, "xxx", 3) != 0
                    && !strcmp(buf2 + ulen + 5, buf))
                {
                    id = atoi(p+1);
                    found = 1;
                    break;
                }
            }

            if (!found)
            {
                ++id;                                /* create a new max ID */
                sprintf(buf2, "%s:%03d:%s\n", uname, id, buf);
                fputs(buf2, fp);
            }
            sprintf(reply, "TADS_%03d.SAV", id);
            retval = OS_AFE_SUCCESS;
            break;
        }
    }

    fclose(fp);
    return(retval);
}

#endif /* USE_DOOR */


/******************************************************************************
* Ports without any special initialization/termination requirements can define
* USE_NULLINIT to pick up the default definitions below.  These do nothing, so
* ports requiring special handling at startup and/or shutdown time must define
* their own versions of these routines.
******************************************************************************/

#ifdef USE_NULLINIT
/* os_init returns 0 for success, 1 for failure.  The arguments are &argc, the
*  address of the count of arguments to the program, and argv, the address of
*  an array of up to 10 pointers to those arguments.  For systems which don't
*  pass a standard command line (such as the Mac Finder), the arguments should
*  be read here using some alternate mechanism (an alert box, for instance),
*  and the values of argc and argv[] updated accordingly.  Note that a maximum
*  of 10 arguments are allowed to be stored in the argv[] array.  The command
*  line itself can be stored in buf, which is a buffer passed by the caller
*  guaranteed to be bufsiz bytes long.
*
*  Unix conventions are followed, so argc is 1 when no arguments are present.
*  The final argument is a prompt string which can be used to ask the user for
*  a command line; its use is not required, but may be desirable for producing
*  a relevant prompt message.  See the Mac implementation for a detailed
*  example of how this mechanism is used.
*/
int os_init(int *argc, char *argv[], const char *prompt,
            char *buf, int bufsiz)
{
    return 0;
}

/*
 *   uninitialize 
 */
void os_uninit(void)
{
}

/* 
 *   os_term should perform any necessary cleaning up, then terminate the
 *   program.  The int argument is a return code to be passed to the
 *   caller, generally 0 for success and other for failure.  
 */
void os_term(int rc)
{
    exit(rc);
}
#endif /* USE_NULLINIT */

/******************************************************************************
* Ports can define USE_NULLPAUSE if no exit pause on ti is needed.
*
* Ports needing an exit pause, and can simply print a message (with os_printf)
* and wait for a key (with os_getc) can define USE_EXPAUSE.
******************************************************************************/

#ifdef USE_NULLPAUSE
void os_expause(void)
{
    /* does nothing */
}
#endif /* USE_NULLPAUSE */

#ifdef USE_EXPAUSE
void os_expause(void)
{
    os_printf( "(Strike any key to exit...)" );
    os_flush();
    os_waitc();
}
#endif /* USE_EXPAUSE */


#ifdef USE_NULLSTAT
/*
 *   USE_NULLSTAT defines a do-nothing version of os_status.
 */
void os_status(int stat)
{
    /* ignore the new status */
}

int os_get_status()
{
    return 0;
}
#endif /* USE_NULLSTAT */

#ifdef USE_NULLSCORE
/*
 *   USE_NULLSCORE defines a do-nothing version of os_score.
 */
void os_score(int cur, int turncount)
{
    /* ignore the score information */
}

void os_strsc(const char *p)
{
    /* ignore */
}
#endif /* USE_NULLSCORE */

#ifdef USE_STATLINE

/* saved main text area column when drawing in long-description area */
static int S_ldesc_savecol;

/* saved main text area column when drawing in status line */
static int S_statline_savecol;

/*
 *   Status line buffer.  Each time we display text to the status line,
 *   we'll keep track of the text here.  This allows us to refresh the
 *   status line whenever we overwrite it (during scrollback mode, for
 *   example).  
 */
static char S_statbuf[OS_MAXWIDTH + 1];

/* pointer to the next free character of the status line buffer */
static char *S_statptr = S_statbuf;


/*
 *  The special run-time version displays a status line and the ldesc before
 *  each command line is read.  To accomplish these functions, we need to
 *  redefine os_gets and os_printf with our own special functions.
 *
 *  Note that os_init may have to modify some of these values to suit, and
 *  should set up the screen appropriately.
 */
void os_status(int stat)
{
    /* check to see what mode we're leaving */
    if (status_mode == 1 && stat != 1)
    {
        /* 
         *   we're leaving status-line mode - restore the main text column
         *   that was in effect before we drew the status line 
         */
        text_lastcol = S_statline_savecol;
    }
    else if (status_mode == 2 && stat != 2)
    {
        /* 
         *   we're leaving long-description mode - clear the rest of the
         *   long-description area after whatever we've written and
         *   restore the last text column 
         */
# ifdef USE_LDESC
#  ifndef USE_GRAPH

        /* keep clearing until we get to the main text area */
        while (ldesc_curline < text_line - 1)
        {
            /* clear this line */
            ossclr(ldesc_curline, ldesc_column, text_line-2, max_column,
                   ldesc_color);

            /* on to the next line */
            ldesc_curline++;
        }
#  endif /* USE_GRAPH */
# endif /* USE_LDESC */

        /* start at the saved column in the text area */
        text_lastcol = S_ldesc_savecol;
    }

    /* switch to the new mode */
    status_mode = stat;

    /* check the mode */
    if (status_mode == 1)
    {
        /* 
         *   we're entering the status line - start writing at the start
         *   of the status line 
         */
        S_statptr = S_statbuf;

        /* 
         *   clear out any previously-saved status line text, since we're
         *   starting a new status line 
         */
        *S_statptr = '\0';

        /* 
         *   remember the current text area display column so that we can
         *   restore it when we finish with the status line 
         */
        S_statline_savecol = text_lastcol;
    }
    else if (status_mode == 2)
    {
        /* 
         *   we're entering long-description mode, so set up to write into
         *   the long-description area 
         */
        
# ifndef USE_FSDBUG
        /* start at the top of the long-description area */
        ldesc_curline = ldesc_line;
# endif /* USE_FSDBUG */

        /* 
         *   remember the text column so that we can restore it when we
         *   leave long-desription mode 
         */
        S_ldesc_savecol = text_lastcol;
    }
}

int os_get_status()
{
    return status_mode;
}

/* Set score to a string value provided by the caller */
void os_strsc(const char *p)
{
    static char lastbuf[135];
    int  i;
    int  x = score_column;

    /* ignore score strings in plain mode, there's no status line */
    if (os_f_plain)
        return;

    /* 
     *   if we have a string, save the new value; if the string pointer is
     *   null, it means that we should redraw the string with the previous
     *   value 
     */
    if (p != 0)
        strcpy(lastbuf, p);
    else
        p = lastbuf;

    /* display enough spaces to right-justify the value */
    for (i = strlen(p) ; i + score_column <= max_column ; ++i)
        ossdsp(sdesc_line, x++, sdesc_color, " ");
    if (x + strlen(p) > (size_t)max_column)
        score_column = x = max_column - strlen(p);
    ossdsp(sdesc_line, x, sdesc_color, p);
    ossdsp(sdesc_line, max_column, sdesc_color, " ");
}

/*
 *   Set the score.  If cur == -1, the LAST score set with a non-(-1)
 *   cur is displayed; this is used to refresh the status line without
 *   providing a new score (for example, after exiting scrollback mode).
 *   Otherwise, the given current score (cur) and turncount are displayed,
 *   and saved in case cur==-1 on the next call.
 */
void os_score(int cur, int turncount)
{
    char buf[20];

    /* check for the special -1 turn count */
    if (turncount == -1)
    {
        /* it's turn "-1" - we're simply redrawing the score */
        os_strsc((char *)0);
    }
    else
    {
        /* format the score */
        sprintf(buf, "%d/%d", cur, turncount);

        /* display the score string */
        os_strsc(buf);
    }
}


# ifdef USE_SCROLLBACK
/*
 *   Pointers for scrollback buffers.
 */
static char *scrbuf, *scrbufp;
static unsigned scrbufl;

void osssbini(unsigned int size)
{
    scrbufl = size;
    if ( !( scrbufp = scrbuf = calloc( scrbufl, 1 )))
    {
        os_printf( "\nSorry, there is no memory for review mode.\n\n" );
    }
}

void osssbdel(void)
{
    if ( scrbuf ) free( scrbuf );
}

static char *ossadvsp(char *p)
{
    p++;
    if ( p >= scrbuf + scrbufl ) p = scrbuf;
    return( p );
}

static char *ossdecsp(char *p)
{
    if ( p == scrbuf ) p = scrbuf + scrbufl;
    return( p-1 );
}

/* add a color control sequence to start of line if necessary */
static void ossaddbold(int sb_text_color)
{
    if (sb_text_color != text_normal_color)
    {
        *scrbufp = 2;              /* store bold code at start of next line */
        scrbufp = ossadvsp(scrbufp);
    }
}

int   os_line_count = 1;             /* count of lines in scrollback buffer */
int   os_top_line;   /* if scrtop if non-null, line # of top line on screen */
static int sb_column;

static void ossaddsb(char *p)
{
    int sb_text_color;

    if ( !scrbufp ) return;

    /*
     *   Copy the text into the screen buffer, respecting the circular
     *   nature of the screen buffer.  If the given text wraps lines,
     *   enter an explicit carriage return into the text to ensure that
     *   users can correctly count lines.
     */
    sb_text_color = text_color;
    while( *p )
    {
        switch(*p)
        {
        case 1:
            sb_text_color = text_normal_color;
            break;
            
        case 2:
            sb_text_color = text_bold_color;
            break;
        }

        /* if overwriting a newline, decrement total line counter */
        if (*scrbufp == '\n') --os_line_count;

        *scrbufp = *p;
        scrbufp = ossadvsp( scrbufp );
        if (*p >= 10) sb_column++;           /* ignore our special controls */
        if ( *p == '\n' )
        {
            ++os_line_count;
            ossaddbold(sb_text_color);
            sb_column = 0;
        }
        if ( *p == '\r' )
        {
            /*
             *   We have a plain carriage return, which indicates that we
             *   should go back to the start of the current line and
             *   overwrite it.  (This is most likely to occur for the
             *   "[More]" prompt.)
             */
            sb_column = 0;
            do
            {
                scrbufp = ossdecsp( scrbufp );      /* back up one character */
            } while ( *scrbufp && *scrbufp != '\n' ); /* go to start of line */
            scrbufp = ossadvsp( scrbufp );             /* move onto the line */
        }
        if ( sb_column > max_column )
        {
            *scrbufp = '\n';
            ++os_line_count;
            sb_column = 0;
            scrbufp = ossadvsp( scrbufp );
            ossaddbold(sb_text_color);
        }
        p++;
    }

    /*
     *   If we've wrapped around into existing screen buffer space,
     *   we must delete it.  Enter nulls up to the next line break.
     */
    if ( *scrbufp != '\0' )
    {
        char *p;
        for ( p=scrbufp ; *p && *p != '\n' ; )
        {
            if (*p == '\n') --os_line_count;             /* deleting a line */
            *p = '\0';
            p = ossadvsp( p );
        }
        if ( *p == '\n' ) --os_line_count, *p = '\0';
    }
}

/*
 *   A slight complication:  if we're not on the
 *   very bottom line, and we don't even have any
 *   text displayed on the bottom line (example:
 *   the input line grew to over 80 characters,
 *   scrolling up a line, but shrunk again due to
 *   deletions), we must add "\n"'s to fill out the
 *   unused space, so the scrollback line counter
 *   doesn't get confused (it counts backwards from
 *   the bottom of the screen, where it assumes we
 *   are right now).  Figure out if the current
 *   buffer will go to the end of the screen.
 *   "i" contains the number of "\n"'s left to do;
 *   each time the buffer overflows the screen
 *   width, we know we are actually using one more
 *   line, so decrement "i".
 *
 *   This routine, ossaddsbe, adds the given text to the screen
 *   buffer, then adds as many blank lines as are necessary to
 *   fill the screen to the bottom line.  buf is the buffer to
 *   add; p is a pointer into the buffer corresponding to screen
 *   position (x,y).
 */
static void ossaddsbe(char *buf, char *p, int x, int y)
{
    int i = max_line - y;                    /* number of blank lines to add */

    for ( ; *p ; p++)
    {
        if ( x > max_column )                 /* we've wrapped to a new line */
        {
            i--;                                     /* one less line to add */
            x=0;                                     /* back to first column */
        }
        if (*p >= 10) ++x;          /* don't count controls in column count */
    }
    ossaddsb( buf );                                /* add the buffer itself */
    while ( i-- ) ossaddsb( "\n" );   /* and any blank lines that are needed */
}

/*
 *   Scrollback mode variables.  These track the current position while
 *   in scrollback mode.
 */
char *scrtop;                         /* first line displayed on the screen */
static char *scrbot;                         /* last line displayed on sreen */
static char *scrlast;                        /* first line of last screenful */

/*
 *   scrdsp  - display a line of text from the scrollback buffer on the
 *   screen.  It is the responsibility of the caller to ensure that the
 *   line on the screen has been cleared beforehand.
 */
static void scrdsp(char *p, int y)
{
    char  buf[135];
    char *q = buf;
    int   color = text_normal_color;
    int   x = text_column;
    int   newcolor;

    while(*p && *p != '\n' && q < buf + sizeof(buf) - 1)
    {
        switch(*p)
        {
        case 2:                                       /* activate bold mode */
            newcolor = text_bold_color;
            goto bold_on_off;
        case 1:                                    /* disactivate bold mode */
            newcolor = text_normal_color;
        bold_on_off:
            *q = '\0';
            ossdsp(y, x, color, buf);
            color = newcolor;
            
            /* go back to start of buffer and proceed */
            q = buf;
            x += strlen(buf);
            break;
            
        default:
            *q++ = *p;
        }
        
        /* advance to next character no matter what happened */
        p = ossadvsp(p);
    }
    *q = '\0';
    ossdsp(y, x, color, buf);
}

/*
 *   scrdspscr - display a screenful of text starting at a given location
 *   in the scrollback buffer.
 */
static void scrdspscr(char *p)
{
    int y;

    ossclr( text_line, text_column, max_line, max_column, text_color );
    for ( y=text_line ; y<=max_line ; y++ )
    {
        scrdsp( p, y );
        while( *p && *p != '\n' )
            p = ossadvsp( p );
        if ( *p ) p = ossadvsp( p );
    }
}

/* move forward by a number of lines, and redisplays the sreen */
static void scrfwd(int n)
{
    if (scrtop == scrlast) return;

    while(n)
    {
        if (scrtop == scrlast) break;
        if (*scrtop == '\n')
        {
            ++os_top_line;
            --n;
            while(*scrbot && *scrbot != '\n') scrbot = ossadvsp(scrbot);
            if (*scrbot) scrbot = ossadvsp(scrbot);
        }
        scrtop = ossadvsp(scrtop);
    }
    scrdspscr(scrtop);
}

/* move back by a number of lines, redisplaying the screen */
static void scrback(int n)
{
    if (!*ossdecsp(scrtop)) return;

    scrtop = ossdecsp(scrtop);           /* back up to end of previous line */
    while(n)                                 /* go until we've seen n lines */
    {
        scrtop = ossdecsp(scrtop);                   /* back up a character */
        if (*scrtop == '\n' || !*scrtop)  /* start of line - back up scrbot */
        {
            --n;                                 /* ... and count this line */
            --os_top_line;
            scrbot = ossdecsp(scrbot);            /* back up scrbot onto \n */

            /* and now find the \n before this line */
            do { scrbot = ossdecsp(scrbot); } while (*scrbot != '\n');
            scrbot = ossadvsp(scrbot);          /* advance to start of line */
        }
        if (!*scrtop) break;                    /* top of buffer - stop now */
    }
    scrtop = ossadvsp(scrtop);                /* advance onto start of line */
    scrdspscr(scrtop);                            /* display the screenfull */
}

/*
 *   scrto moves to a selected line
 */
void scrto(int lin)
{
    if (lin > os_top_line)
        scrfwd(lin - os_top_line);
    else if (lin < os_top_line)
        scrback(os_top_line - lin);
}

/*
 *   scrpgup scrolls back a page
 */
void scrpgup(void)
{
    scrback(max_line - text_line);
}

/*
 *   scrpgdn scrolls forward a page
 */
void scrpgdn(void)
{
    scrfwd(max_line - text_line);
}

/*
 *   scrlnup scrolls up one line
 */
void scrlnup(void)
{
    if ( *ossdecsp( scrtop ) == '\0' ) return;

    scrtop = ossdecsp( scrtop );
    scrbot = ossdecsp( scrbot );
    do { scrtop = ossdecsp( scrtop ); } while ( *scrtop && *scrtop != '\n' );
    do { scrbot = ossdecsp( scrbot ); } while ( *scrbot && *scrbot != '\n' );
    scrtop = ossadvsp( scrtop );
    scrbot = ossadvsp( scrbot );
    --os_top_line;

    ossscu( text_line, text_column, max_line, max_column, text_color );
    scrdsp( scrtop, text_line );
}

/*
 *   scrlndn scrolls down one line
 */
void scrlndn(void)
{
    if ( scrtop == scrlast ) return;

    while( *scrtop != '\n' ) scrtop = ossadvsp( scrtop );
    while( *scrbot != '\n' && *scrbot ) scrbot = ossadvsp( scrbot );
    scrtop = ossadvsp( scrtop );
    if ( *scrbot ) scrbot = ossadvsp( scrbot );
    ++os_top_line;

    ossscr( text_line, text_column, max_line, max_column, text_color );
    scrdsp( scrbot, max_line );
}

/* redraw the screen's text region */
void os_redraw(void)
{
    int   y = max_line;
    char *p;

#ifdef USE_GRAPH
    os_drawcontrols();
#endif /* USE_GRAPH */

    ossclr(text_line, text_column, max_line, max_column, text_color);
    p = scrbufp;
    while (y >= text_line)
    {
        p = ossdecsp(p);
        if (*p == '\0')
        {
            int old_text_line = text_line;
            text_line = y;
            p = ossadvsp(p);
            scrdspscr(p);
            text_line = old_text_line;
            return;
        }
        if (*p == '\n') --y;
    }
    p = ossadvsp(p);
    scrdspscr(p);
}

/* vertically resize the text window, redisplaying text */
int ossresize(int newtop, int newbot)
{
    /* a size of -1 indicates no change; note each changed value */
    if (newtop != -1)
        text_line = newtop;
    if (newtop != -1)
        max_line = newbot;

    /* notify the portable formatting code of the change */
    G_os_pagelength = max_line - text_line - 1;

    /* redraw the screen at the new size */
    os_redraw();

    /* success */
    return 0;
}

/*
 *   osssbmode toggles scrollback mode.  Once in scrollback mode, calls
 *   can be made to osssbup, osssbdn, osssbpgup, and osspgdn to scroll
 *   through the saved text.  We also put up a scrollback-mode version of
 *   the status line, showing keys that should be used for scrollback.
 *   When the user wants to exit scrollback mode, this routine should
 *   be called again.
 *
 *   Returns:  0 if scrollback mode is entered/exited successfully, 1 if
 *   an error occurs.  Note that it may not be possible to enter scrollback
 *   mode, since insufficient saved text may have been accumulated.
 */
int osssbmode(int mode_line)
{
    /* if there's no buffer, we can't enter scrollback mode */
    if (scrbufp == 0)
        return 1;

    /* 
     *   if we're not in scrollback mode, enter scrollback mode;
     *   otherwise, return to normal mode 
     */
    if (scrtop == 0)
    {
        /*
         *   Enter scrollback mode.  Figure out what scrtop should be, and
         *   put up the scrollback status line.  If insufficient saved text
         *   is around for scrollback mode, return 1.
         */
        int  y = max_line;
        char buf[135];
        int  i;

        scrtop = scrbufp;                          /* start at end of buffer */
        os_top_line = os_line_count;      /* start at line count for buffer */
        while (y >= text_line)            /* keep going until top of screen */
        {
            /* back up one byte */
            scrtop = ossdecsp(scrtop);

            /* have we reached the start of the buffer? */
            if (*scrtop == '\0')
            {
                /* 
                 *   there wasn't enough saved text, so abort scrollback
                 *   mode and return failure 
                 */
                scrtop = 0;
                return 1;
            }

            /* if we reached a newline, count it */
            if (*scrtop == '\n')
            {
                --y;
                --os_top_line;
            }
        }

        /* move forward over '\n' to start of line */
        scrtop = ossadvsp(scrtop);

        /* remember where current screen starts */
        scrlast = scrtop;

        /*
         *   Note the last line on the screen as well.  Just find the
         *   beginning of the last line currently in the buffer.
         */
        scrbot = ossdecsp(scrbufp);
        while (*scrbot != '\n')
            scrbot = ossdecsp(scrbot);
        scrbot = ossadvsp(scrbot);

        /* construct the appropriate status line, and display it */
        if (mode_line)
        {
            strcpy(buf, OS_SBSTAT);
            for (i = strlen(buf) ; i < max_column ; buf[i++] = ' ') ;
            buf[i] = '\0';
            ossdsp(sdesc_line, sdesc_column+1, sdesc_color, buf);
        }

        /* successfully entered scrollback mode */
        return 0;
    }
    else
    {
        /*
         *   Exit scrollback mode.  Show the last page of text, and put
         *   up the normal status bar.  Also clear our scrollback mode
         *   variables.
         */

        if (scrlast != scrtop)
            scrdspscr(scrlast);

# ifdef USE_LDESC
#  ifndef USE_FSDBUG
#   define STAT_TYPE 1
#  endif /* USE_FSDBUG */
# endif /* USE_LDESC */
# ifndef STAT_TYPE
#  define STAT_TYPE 0
# endif /* STAT_TYPE */
        if (mode_line)
        {
            /* display the last status line buffer */
            ossdspn(sdesc_line, sdesc_column, sdesc_color, " ");
            ossdspn(sdesc_line, sdesc_column + 1, sdesc_color, S_statbuf);

            /* 
             *   refresh the right-hand side with the score part - do this
             *   by drawing the score with a special turn counter of -1 to
             *   indicate that we just want to refresh the previous score
             *   value 
             */
            os_score(-1, -1);
        }
        scrtop = 0;
        return( 0 );
    }
}

/*
 *   ossdosb runs a scrollback session.  When entered, osssbmode()
 *   must already have been called.  When we're done, we'll be out
 *   of scrollback mode.
 */
static void ossdosb(void)
{
    for ( ;; )
    {
        if (!os_getc())
        {
            switch(os_getc())
            {
            case CMD_SCR:
            case CMD_KILL:         /* leave scrollback via 'escape' as well */
            case CMD_EOF:                            /* stop on end of file */
                osssbmode(1);
                return;
            case CMD_UP:
                scrlnup();
                break;
            case CMD_DOWN:
                scrlndn();
                break;
            case CMD_PGUP:
                scrpgup();
                break;
            case CMD_PGDN:
                scrpgdn();
                break;
            }
        }
    }
}

# else /* USE_SCROLLBACK */
static void ossaddsb(char *p)
{
    /* We're not saving output - do nothing */
}
# endif /* USE_SCROLLBACK */

/* display with no highlighting - intercept and ignore highlight codes */
void ossdspn(int y, int x, int color, char *p)
{
    char *q;
    
    for (q = p ; *q ; ++q)
    {
        switch(*q)
        {
        case 1:                                          /* set mode normal */
        case 2:                                       /* set mode highlight */
            *q = '\0';
            ossdsp(y, x, color, p);
            x += strlen(p);
            p = q + 1;
            break;

        default:
            break;
        }
    }
    if (q != p) ossdsp(y, x, color, p);             /* display last portion */
}

/* display with highlighting enabled */
int ossdsph(int y, int x, int color, char *p)
{
    char *q;
    int   newcolor;
    int   len;

    /* get length of entire string */
    len = strlen(p);
    
    for (q = p ; *q ; ++q)
    {
        switch(*q)
        {
        case 1:                                          /* set mode normal */
            newcolor = text_normal_color;
            goto bold_on_off;
        case 2:                                       /* set mode highlight */
            newcolor = text_bold_color;
        bold_on_off:
            *q = '\0';
            ossdsp(y, x, text_color, p);
            x += strlen(p);
            text_color = newcolor;
            p = q + 1;

            /* don't count the switch character in the length */
            --len;
            break;

        default:
            break;
        }
    }

    /* display the last portion of the line if anything's left */
    if (q != p)
        ossdsp(y, x, text_color, p);

    /* return the length */
    return len;
}

/*
 *   Set the terminal into 'plain' mode: disables status line,
 *   scrollback, command editing.
 */
void os_plain(void)
{
    /* set the 'plain' mode flag */
    os_f_plain = 1;
}

void os_printf(const char *fmt, ...)
{
    va_list argptr;

    va_start(argptr, fmt);
    os_vprintf(fmt, argptr);
    va_end(argptr);
}

void os_vprintf(const char *fmt, va_list argptr)
{
    char buf[256];

    /* apply the formatting */
    vsprintf(buf, fmt, argptr);

    /* 
     *   save the output in the scrollback buffer if we're displaying to
     *   the main text area (status_mode == 0) and we're not in plain
     *   stdio mode (in which case there's no scrollback support) 
     */
    if (status_mode == 0 && !os_f_plain)
        ossaddsb(buf);

    /* determine what to do based on the status mode */
    switch(status_mode)
    {
        case 2:                                        /* ldesc display mode */
# ifdef USE_LDESC
            /* ignore the ldesc display in plain mode */
            if (os_f_plain)
                break;

            /* 
             *   if we've already filled the available vertical area,
             *   ignore this text
             */
            if (ldesc_curline >= text_line-1)
                break;
            
            /* otherwise, fall through to normal display code */
# else /* USE_LDESC */
            break;                       /* don't allow ldesc display at all */
# endif /* USE_LDESC */
            
        case 0:                                           /* main text area */
        {
            char *p;

            for ( p=buf ; *p ; )
            {
                char *p1;

                for (p1 = p ; *p1 && *p1 != '\n' && *p1 != '\r' ; p1++);
                if (*p1 == '\n' || *p1 == '\r')
                {
                    int c = *p1;

                    *p1 = '\0';
                    if (status_mode == 0)
                    {
                        if (os_f_plain)
                        {
                            char cbuf[2];
                            
                            fputs(p, stdout);
                            cbuf[0] = c;
                            cbuf[1] = '\0';
                            fputs(cbuf, stdout);
                        }
                        else
                        {
                            ossdsph(max_line, text_lastcol, text_color, p);
                            if (c == '\n')
                                ossscr(text_line, text_column, max_line,
                                       max_column, text_color);
                        }
                    }
# ifdef USE_LDESC
                    else
                    {
                        ossdspn(ldesc_curline, text_lastcol, ldesc_color, p);
                        ossclr(ldesc_curline, text_lastcol + strlen(buf),
                               ldesc_curline, max_column, ldesc_color);
                        ++ldesc_curline;
                    }
# endif /* USE_LDESC */
                    text_lastcol = text_column;
                    p = p1 + 1;
                }
                else
                {
                    int len;
                        
                    if (status_mode == 0)
                    {
                        if (os_f_plain)
                        {
                            len = strlen(p);
                            fputs(p, stdout);
                        }
                        else
                        {
                            len = ossdsph(max_line, text_lastcol,
                                          text_color, p);
                        }
                    }
                    else
                    {
                        len = strlen(p);
                        ossdspn(ldesc_curline++, text_lastcol, ldesc_color,p);
                    }

                    text_lastcol += len;
                    p = p1;
                }
                if (status_mode == 0 && !os_f_plain)
                    ossloc(max_line, text_lastcol);
            }
            break;
        }
        case 1:                                          /* status-line mode */
        {
            int   i;
            char *p;

            /* ignore status line in 'plain' mode */
            if (os_f_plain)
                break;

            /* check for a newline */
            for (p = buf ; *p == '\n' ; p++);

            /* 
             *   if we found a newline, truncate at the newline - the
             *   status line only goes for a single line 
             */
            if ((i = strlen(p)) != 0 && p[i-1] == '\n')
                p[--i] = '\0';

            /* add spaces up to the score location */
            while (i < score_column-1)
                buf[i++] = ' ';

            /* null-terminate the buffer */
            buf[i] = '\0';
            
#ifndef USE_GRAPH
            /* display a leading space */
            ossdspn(sdesc_line, sdesc_column, sdesc_color, " ");
#endif /* USE_GRAPH */

            /* display the string */
            ossdspn(sdesc_line, sdesc_column + 1, sdesc_color, buf);

            /* 
             *   add this text to our private copy, so that we can refresh
             *   the display later if necessary 
             */
            for (p = buf ;
                 *p != '\0' && S_statptr < S_statbuf + sizeof(S_statbuf) ; )
                *S_statptr++ = *p++;

            /* 
             *   null-terminate our private copy at the current position,
             *   in case we don't add anything later 
             */
            *S_statptr = '\0';

            /* 
             *   switch to the ldesc portion (if we support ldesc mode; if
             *   not, this will suppress any subsequent output until we
             *   return to normal main-text mode) 
             */
            os_status(2);
            break;
        }
    }
}

void os_flush(void)
{
    /* we don't buffer our output, so there's nothing to do here */
}

# ifdef USE_HISTORY
/*
 *   For command line history, we must have some buffer space to store
 *   past command lines.  We will use a circular buffer:  when we move
 *   the pointer past the end of the buffer, it wraps back to the start
 *   of the buffer.  A "tail" indicates the oldest line in the buffer;
 *   when we need more room for new text, we advance the tail and thereby
 *   lose the oldest text in the buffer.
 */
static unsigned char *histbuf = 0;
static unsigned char *histhead = 0;
static unsigned char *histtail = 0;

/*
 *   ossadvhp advances a history pointer, and returns the new pointer.
 *   This function takes the circular nature of the buffer into account
 *   by wrapping back to the start of the buffer when it hits the end.
 */
uchar *ossadvhp(uchar *p)
{
    if (++p >= histbuf + HISTBUFSIZE)
        p = histbuf;
    return p;
}

/*
 *   ossdechp decrements a history pointer, wrapping the pointer back
 *   to the top of the buffer when it reaches the bottom.
 */
uchar *ossdechp(uchar *p)
{
    if (p == histbuf)
        p = histbuf + HISTBUFSIZE;
    return p - 1;
}

/*
 *  osshstcpy copies from a history buffer into a contiguous destination
 *  buffer, wrapping the history pointer if need be.  One null-terminated
 *  string is copied.
 */
void osshstcpy(uchar *dst, uchar *hst)
{
    while( *hst )
    {
        *dst++ = *hst;
        hst = ossadvhp( hst );
    }
    *dst = '\0';
}

/*
 *   ossprvcmd returns a pointer to the previous history command, given
 *   a pointer to a history command.  It returns a null pointer if the
 *   given history command is the first in the buffer.
 */
uchar *ossprvcmd(uchar *hst)
{
    if (hst == histtail)
        return 0;                              /* no more previous commands */
    hst = ossdechp( hst );                                  /* back onto nul */
    do
    {
        hst = ossdechp( hst );                        /* back one character */
    } while (*hst && hst != histtail);         /* go until previous command */
    if (*hst == 0)
        hst = ossadvhp(hst);                         /* step over null byte */
    return hst;
}

/*
 *   ossnxtcmd returns a pointer to the next history command, given
 *   a pointer to a history command.  It returns a null pointer if the
 *   given command is already past the last command.
 */
uchar *ossnxtcmd(uchar *hst)
{
    if ( hst == histhead ) return( 0 );         /* past the last one already */
    while( *hst ) hst = ossadvhp( hst );             /* scan forward to null */
    hst = ossadvhp( hst );                /* scan past null onto new command */
    return( hst );
}
# endif /* USE_HISTORY */

int  osqstate;
unsigned char osqbuf[64];

/*
 *   Read a command from the keyboard.  This implementation provides
 *   command editing and history. 
 */
uchar *os_gets(unsigned char *buf, size_t bufl)
{
    unsigned char *p = buf;
    unsigned char *eol = buf;
    unsigned char *eob = buf + bufl;
    int   x = text_lastcol;
    int   y = max_line;
    int   origx = x;
# ifdef USE_HISTORY
    unsigned char *curhist = histhead;
    unsigned char  savbuf[135];
# endif /* USE_HISTORY */
    extern int dbgactive;

# ifdef USE_HISTORY
    /* allocate the history buffer if it's not already allocated */
    if (histbuf == 0)
    {
        histbuf = (unsigned char *)osmalloc(HISTBUFSIZE);
        histhead = histtail = histbuf;
        curhist = histhead;
    }
# endif /* USE_HISTORY */

    /* disable command editing in 'plain' mode and just use stdio input */
    if (os_f_plain)
        return (uchar *)gets((char *)buf);

#  ifdef USE_LDESC
    text_lastcol = 0;
#  endif /* USE_LDESC */

    /* move to the starting position for the command */
    ossloc(y, x);

    /* we're ready to start accepting commands */
    osqstate = 1;

    /* clear out the buffer */
    buf[0] = '\0';

    /* process keystrokes until we're done entering the command */
    for ( ;; )
    {
        unsigned char c;
        
        /* if there's a queued command, use it */
        if (osqstate == 2)
        {
            c = CMD_QUEUE;
            goto replace_line;
        }

        /* move to the proper position on the screen */
        ossloc(y, x);

        /* get a keystroke */
        c = os_getc();

        /* check the character we got */
        switch(c)
        {
        case 8:
            if (p > buf)
            {
                uchar *q;
                uchar  tmpbuf[2];
                int    thisx, thisy;
                
                for (q = --p ; q < eol ; ++q)
                    *q = *(q+1);
                --eol;
                if (--x < 0)
                {
                    x = max_column;
                    --y;
                }
                *eol = ' ';
                thisx = x;
                thisy = y;
                for (q = p, tmpbuf[1] = '\0' ; q <= eol ; ++q)
                {
                    tmpbuf[0] = *q;
                    ossdsp(thisy, thisx, text_color, (char *)tmpbuf);
                    if (++thisx > max_column)
                    {
                        thisx = 0;
                        thisy++;
                    }
                }
                *eol = '\0';
            }
            break;

        case 13:
        return_line:
            /*
             *   Scroll the screen to account for the carriage return,
             *   position the cursor at the end of the new line, and
             *   null-terminate the line.  
             */
            *eol = '\0';
            while(p != eol)
            {
                ++p;
                if (++x > max_column)
                {
                    ++y;
                    x = 0;
                }
            }
            ossloc(y, x);
            if (y == max_line)
                ossscr(text_line, text_column, max_line, max_column,
                       text_color);
            text_lastcol = 0;

# ifdef USE_HISTORY
            /*
             *   Save the line in our history buffer.  If we don't have
             *   enough room, lose some old text by advancing the tail
             *   pointer far enough.  Don't save it if it's a blank line,
             *   though, or if it duplicates the most recent previous
             *   command.
             */
            if (strlen((char *)buf) != 0)
            {
                uchar *q;
                int    advtail;
                int    saveflag = 1;         /* assume we will be saving it */
                
                if (q = ossprvcmd(histhead))
                {
                    uchar *p = buf;
                    
                    while (*p == *q && *p != '\0' && *q != '\0')
                    {
                        ++p;
                        q = ossadvhp(q);
                    }
                    if (*p == *q)           /* is this a duplicate command? */
                        saveflag = 0;               /* if so, don't save it */
                }

                if (saveflag)
                {
                    for (q = buf, advtail = 0 ; q <= eol ; ++q)
                    {
                        *histhead = *q;
                        histhead = ossadvhp(histhead);
                        if (histhead == histtail)
                        {
                            histtail = ossadvhp(histtail);
                            advtail = 1;
                        }
                    }
                    
                    /*
                     *   If we have encroached on space that was already
                     *   occupied, throw away the entire command we have
                     *   partially trashed; to do so, advance the tail
                     *   pointer to the next null byte.  
                     */
                    if (advtail)
                    {
                        while(*histtail)
                            histtail = ossadvhp(histtail);
                        histtail = ossadvhp(histtail);
                    }
                }
            }
# endif /* USE_HISTORY */

            /*
             *   Finally, copy the buffer to the screen save buffer (if
             *   applicable), and return the contents of the buffer.  Note
             *   that we add an extra carriage return if we were already
             *   on the max_line, since we scrolled the screen in this
             *   case; otherwise, ossaddsbe will add all the blank lines
             *   that are necessary.  
             */
            ossaddsbe((char *)buf, (char *)p, x, y);
            if (y == max_line)
                ossaddsb("\n");
            osqstate = 0;

            /* return success */
            return buf;

        case 0:
            /* extended key code - get the second half of the code */
            switch(c = os_getc())
            {
# ifdef USE_SCROLLBACK
            case CMD_SCR:
                {
                    char *savbufp = scrbufp;
                    
                    /*
                     *   Add the buffer, plus any blank lines, to the
                     *   screen buffer, filling the screen to the bottom.  
                     */
                    ossaddsbe((char *)buf, (char *)p, x, y);
                    
                    if (!osssbmode(1))
                        ossdosb();
                    scrbufp = savbufp;
                    *scrbufp = '\0';
                    sb_column = origx;
                    break;
                }
# endif /* USE_SCROLLBACK */

            case CMD_LEFT:
                if (p > buf)
                {
                    --p;
                    --x;
                    if (x < 0)
                    {
                        x = max_column;
                        --y;
                    }
                }
                break;

            case CMD_WORD_LEFT:
                if (p > buf)
                {
                    --p;
                    --x;
                    if (x < 0)
                        x = max_column, --y;
                    while (p > buf && isspace(*p))
                    {
                        --p, --x;
                        if (x < 0)
                            x = max_column, --y;
                    }
                    while (p > buf && !isspace(*(p-1)))
                    {
                        --p, --x;
                        if (x < 0)
                            x = max_column, --y;
                    }
                }
                break;

            case CMD_RIGHT:
                if (p < eol)
                {
                    ++p;
                    ++x;
                    if (x > max_column)
                    {
                        x = 0;
                        ++y;
                    }
                }
                break;

            case CMD_WORD_RIGHT:
                while (p < eol && !isspace(*p))
                {
                    ++p, ++x;
                    if (x > max_column)
                        x = 0, ++y;
                }
                while (p < eol && isspace(*p))
                {
                    ++p, ++x;
                    if (x > max_column)
                        x = 0, ++y;
                }
                break;

            case CMD_DEL:
                if (p < eol)
                {
                    uchar *q;
                    uchar  tmpbuf[2];
                    int    thisx = x, thisy = y;
                    
                    for (q = p ; q < eol ; ++q)
                        *q = *(q+1);
                    --eol;
                    *eol = ' ';
                    for (q = p, tmpbuf[1] = '\0' ; q <= eol ; ++q)
                    {
                        tmpbuf[0] = *q;
                        ossdsp(thisy, thisx, text_color, (char *)tmpbuf);
                        if (++thisx > max_column)
                        {
                            thisx = 0;
                            ++thisy;
                        }
                    }
                    *eol = '\0';
                }
                break;
#ifdef UNIX               

            case CMD_WORDKILL:
                {
                    uchar *q;
                    uchar tmpbuf[2];
                    int  thisx, thisy;
                    
                    /* remove spaces preceding word */
                    while (p >= buf && *p <= ' ')
                    {
                        for (q = --p ; q < eol ; ++q)
                            *q = *(q+1);
                        --eol;
                        
                        if (--x < 0)
                        {
                            x = max_column;
                            --y;
                        }
                        *eol = ' ';
                        thisx = x;
                        thisy = y;
                    }
                    
                    /* remove previous word (i.e., until we get a space) */
                    while (p >= buf && *p > ' ')
                    {
                        for (q = --p; q < eol; ++q)
                            *q = *(q+1);
                        --eol;
                        
                        if (--x < 0)
                        {
                            x = max_column;
                            --y;
                        }
                        *eol = ' ';
                        thisx = x;
                        thisy = y;
                    }
                    
                    for (q = p, tmpbuf[1] = '\0' ; q <= eol ; ++q)
                    {
                        tmpbuf[0] = *q;
                        ossdsp(thisy, thisx, text_color, tmpbuf);
                        if (++thisx > max_column)
                        {
                            thisx = 0;
                            thisy++;
                        }
                    }
                    *eol = '\0';
                    break;
                }
#endif /* UNIX */

            case CMD_KILL:
            case CMD_HOME:
# ifdef USE_HISTORY
            case CMD_UP:
            case CMD_DOWN:
            replace_line:
                if (c == CMD_UP && !ossprvcmd(curhist))
                    break;                /* no more history - ignore arrow */
                if (c == CMD_DOWN && !ossnxtcmd(curhist))
                    break;                /* no more history - ignore arrow */
                if (c == CMD_UP && !ossnxtcmd(curhist))
                {
                    /* first Up arrow - save current buffer */
                    strcpy((char *)savbuf, (char *)buf);
                }
# endif /* USE_HISTORY */
                while(p > buf)
                {
                    --p;
                    if (--x < 0)
                    {
                        x = max_column;
                        --y;
                    }
                }
                if (c == CMD_HOME)
                    break;

                /*
                 *   We're at the start of the line now; fall through for
                 *   KILL, UP, and DOWN to the code which deletes to the
                 *   end of the line.  
                 */
            case CMD_DEOL:
                if (p < eol)
                {
                    uchar *q;
                    int    thisx = x, thisy = y;
                    
                    for (q = p ; q < eol ; ++q)
                    {
                        ossdsp(thisy, thisx, text_color, " ");
                        if (++thisx > max_column)
                        {
                            thisx = 0;
                            ++thisy;
                        }
                    }
                    eol = p;
                    *p = '\0';
                }
# ifdef USE_HISTORY
                if (c == CMD_UP)
                {
                    curhist = ossprvcmd(curhist);
                    osshstcpy(buf, curhist);
                }
                else if (c == CMD_DOWN)
                {
                    if (!ossnxtcmd(curhist))
                        break;                                   /* no more */
                    curhist = ossnxtcmd(curhist);
                    if (ossnxtcmd(curhist))           /* on a valid command */
                        osshstcpy(buf, curhist);           /* ... so use it */
                    else
                    {
                        /* no more history - restore original line */
                        strcpy((char *)buf, (char *)savbuf);
                    }
                }
                if ((c == CMD_UP || c == CMD_DOWN)
                    && strlen((char *)buf) != 0)
                {
                    char tmpbuf[2];
                    
                    tmpbuf[1] = '\0';
                    eol = buf + strlen((char *)buf);
                    while(p < eol)
                    {
                        tmpbuf[0] = *p++;
                        ossdsp(y, x, text_color, tmpbuf);
                        if (++x > max_column)
                        {
                            x = 0;
                            if (y == max_line)
                                ossscr(text_line, text_column,
                                       max_line, max_column, text_color);
                            else
                                ++y;
                        }
                    }
                }
# endif /* USE_HISTORY */
                if (c == CMD_QUEUE)
                {
                    strcpy((char *)buf, (char *)osqbuf);
                    ossdsp(y, x, text_color, (char *)buf);
                    p = eol = buf;
                    eol += strlen((char *)eol);
                    goto return_line;
                }
                break;
            case CMD_END:
                while (p < eol)
                {
                    ++p;
                    if (++x > max_column)
                    {
                        x = 0;
                        ++y;
                    }
                }
                break;

            case CMD_EOF:
                /* on end of file, return null */
                return 0;
            }
            break;

        default:
            if (c >= ' ' && eol < eob)
            {
                if (p != eol)
                {
                    uchar *q;
                    int    thisy = y, thisx = x;
                    uchar  tmpbuf[2];
                    
                    for (q = ++eol ; q > p ; --q)
                        *q = *(q-1);
                    *p = c;
                    for (q = p++, tmpbuf[1] = '\0' ; q < eol ; ++q)
                    {
                        tmpbuf[0] = *q;
                        ossdsp(thisy, thisx, text_color, (char *)tmpbuf);
                        ++thisx;
                        if (thisx > max_column)
                        {
                            thisx = 0;
                            if (thisy == max_line)
                            {
                                --y;
                                ossscr(text_line, text_column, max_line,
                                       max_column, text_color);
                            }
                            else
                                ++thisy;
                        }
                    }
                    if (++x > max_column)
                    {
                        ++y;
                        x = 0;
                    }
                }
                else
                {
                    *p++ = c;
                    *p = '\0';
                    ++eol;
                    ossdsp(y, x, text_color, (char *)p-1);
                    if (++x > max_column)
                    {
                        x = 0;
                        if (y == max_line)
                            ossscr(text_line, text_column, max_line,
                                   max_column, text_color);
                        else
                            ++y;
                    }
                }
            }
            break;
        }
    }
}

#else /* USE_STATLINE */

#endif /* USE_STATLINE */

#ifdef STD_OS_HILITE

#ifdef RUNTIME
/*
 *   Return character to insert to turn hilighting on (mode==2) or off
 *   (mode==1); a return of zero indicates no highlighting is supported.
 */
int os_hilite(int mode)
{
    return(os_f_plain ? 0 : mode);   /* just use same character as they use */
}
#else /* RUNTIME */
int os_hilite(int mode)
{
    return(0);                                 /* no highlighting supported */
}
#endif /* RUNTIME */

#endif /* STD_OS_HILITE */

#ifdef STD_OSCLS

/* clear the screen - loses the scrollback buffer */
void oscls(void)
{
#ifdef RUNTIME
    if (os_f_plain) return;
    scrbufp = scrbuf;
    scrtop = scrbot = scrlast = 0;
    memset(scrbuf, 0, (size_t)scrbufl);
    os_redraw();
#endif
}

#endif /* STD_OSCLS */

/*
 *   Simple implementation of os_get_sysinfo.  This can be used for any
 *   non-HTML version of the system, since all sysinfo codes currently
 *   pertain to HTML features.  Note that new sysinfo codes may be added
 *   in the future which may be relevant to non-html versions, so the
 *   sysinfo codes should be checked from time to time to ensure that new
 *   codes relevant to this system version are handled correctly here.  
 */
int os_get_sysinfo(int code, void *param, long *result)
{
    switch(code)
    {
    case SYSINFO_HTML:
    case SYSINFO_JPEG:
    case SYSINFO_PNG:
    case SYSINFO_WAV:
    case SYSINFO_MIDI:
    case SYSINFO_WAV_MIDI_OVL:
    case SYSINFO_WAV_OVL:
    case SYSINFO_PREF_IMAGES:
    case SYSINFO_PREF_SOUNDS:
    case SYSINFO_PREF_MUSIC:
    case SYSINFO_PREF_LINKS:
        /* 
         *   we don't support any of these features - set the result to 0
         *   to indicate this 
         */
        *result = 0;

        /* return true to indicate that we recognized the code */
        return TRUE;

    default:
        /* we don't recognize other codes */
        return FALSE;
    }
}

/*
 *   Set the saved-game extension.  Most platforms don't need to do
 *   anything with this information, and in fact most platforms won't even
 *   have a way of letting the game author set the saved game extension,
 *   so this trivial implementation is suitable for most systems.
 *   
 *   The purpose of setting a saved game extension is to support platforms
 *   (such as Windows) where the filename suffix is used to associate
 *   document files with applications.  Each stand-alone executable
 *   generated on such platforms must have a unique saved game extension,
 *   so that the system can associate each game's saved position files
 *   with that game's executable.  
 */
void os_set_save_ext(const char *ext)
{
    /* ignore the setting */
}


/* ------------------------------------------------------------------------ */
/*
 *   Set the game title.  Most platforms have no use for this information,
 *   so they'll just ignore it.  This trivial implementation simply
 *   ignores the title. 
 */
#ifdef USE_NULL_SET_TITLE

void os_set_title(const char *title)
{
    /* ignore the information */
}

#endif /* USE_NULL_SET_TITLE */

