#include "Operand.hpp"

#include "String.hpp"

namespace SoftWire
{
	const Specifier Specifier::specifierSet[] =
	{
		{UNKNOWN,	""},

		{NEAR,		"NEAR"},
		{SHORT,		"SHORT"},
	//	{FAR,		"FAR"},

		{BYTE,		"BYTE"},
		{WORD,		"WORD"},
		{DWORD,		"DWORD"},
	//	{TWORD,		"TWORD"},
		{QWORD,		"QWORD"},
		{MMWORD,	"MMWORD"},
		{XMMWORD,	"XMMWORD"},
		{XWORD,		"XWORD"},
		{OWORD,		"OWORD"},

		{PTR,		"PTR"}
	};

	Specifier::Type Specifier::scan(const char *string)
	{
		if(string)
		{
			for(int i = 0; i < sizeof(specifierSet) / sizeof(Specifier); i++)
			{
				if(stricmp(string, specifierSet[i].notation) == 0)
				{
					return specifierSet[i].type;
				}		
			}
		}

		return UNKNOWN;
	}

	bool Operand::operator==(Operand &op)
	{
		return type == op.type &&
		       baseReg == op.baseReg &&
			   indexReg == op.indexReg &&
			   scale == op.scale &&
			   displacement == op.displacement;
	}

	bool Operand::operator!=(Operand &op)
	{
		return type != op.type ||
		       baseReg != op.baseReg ||
			   indexReg != op.indexReg ||
			   scale != op.scale ||
			   displacement != op.displacement;
	}

	bool Operand::isSubtypeOf(Type type, Type baseType)
	{
		return (type & baseType) == type;
	}

	bool Operand::isSubtypeOf(Type baseType) const
	{
		return isSubtypeOf(type, baseType);
	}

	const char *Operand::string() const
	{
		static char string[256];

		if(isVoid(type))
		{
			return 0;
		}
		else if(isImm(type))
		{
			if(reference)
			{
				return reference;
			}
			else
			{
				if(value <= 127 && value >= -128)
				{
					snprintf(string, 255, "0x%0.2X", value & 0xFF);
				}
				else if(value <= 32767 && value -32768)
				{
					snprintf(string, 255, "0x%0.4X", value & 0xFFFF);
				}
				else
				{
					snprintf(string, 255, "0x%0.8X", value);
				}
			}
		}
		else if(isReg(type))
		{
			return regName();
		}
		else if(isMem(type))
		{
			switch(type)
			{
			case MEM8:
				snprintf(string, 255, "byte ptr [");
				break;
			case MEM16:
				snprintf(string, 255, "word ptr [");
				break;
			case MEM32:
				snprintf(string, 255, "dword ptr [");
				break;
			case MEM64:
				snprintf(string, 255, "qword ptr [");
				break;
			case MEM128:
				snprintf(string, 255, "xmmword ptr [");
				break;
			case MEM:
			default:
				snprintf(string, 255, "byte ptr [");
			}

			if(baseReg != Encoding::REG_UNKNOWN)
			{
				snprintf(string, 255, "%s%s", string, regName());

				if(indexReg != Encoding::REG_UNKNOWN)
				{
					snprintf(string, 255, "%s+", string);
				}
			}

			if(indexReg != Encoding::REG_UNKNOWN)
			{
				snprintf(string, 255, "%s%s", string, indexName());
			}

			switch(scale)
			{
			case 0:
			case 1:
				break;
			case 2:
				snprintf(string, 255, "%s*2", string);
				break;
			case 4:
				snprintf(string, 255, "%s*4", string);
				break;
			case 8:
				snprintf(string, 255, "%s*8", string);
				break;
			default:
				throw INTERNAL_ERROR;
			}

			if(displacement)
			{
				if(baseReg != Encoding::REG_UNKNOWN ||
				   indexReg != Encoding::REG_UNKNOWN)
				{
					snprintf(string, 255, "%s+", string);
				}

				if(reference)
				{
					snprintf(string, 255, "%s%s", string, reference);
				}
				else
				{
					if(displacement <= 32767 && displacement >= -32768)
					{
						snprintf(string, 255, "%s%d", string, displacement);
					}
					else
					{					
						snprintf(string, 255, "%s0x%0.8X", string, displacement);
					}
				}
			}

			snprintf(string, 255, "%s]", string);
		}
		else if(isStr(type))
		{
			return notation;
		}
		else
		{
			throw INTERNAL_ERROR;
		}

		return strlwr(string);
	}

	bool Operand::isVoid(Type type)
	{
		return type == VOID;
	}

	bool Operand::isImm(Type type)
	{
		return (type & IMM) == type;
	}

	bool Operand::isReg(Type type)
	{
		return (type & REG) == type;
	}

	bool Operand::isMem(Type type)
	{
		return (type & MEM) == type;
	}

	bool Operand::isR_M(Type type)
	{
		return (type & R_M) == type;
	}

	bool Operand::isStr(Type type)
	{
		return (type & STR) == type;
	}

	bool Operand::isVoid(const Operand &operand)
	{
		return isVoid(operand.type);
	}

	bool Operand::isImm(const Operand &operand)
	{
		return isImm(operand.type);
	}

	bool Operand::isReg(const Operand &operand)
	{
		return isReg(operand.type);
	}

	bool Operand::isMem(const Operand &operand)
	{
		return isMem(operand.type);
	}

	bool Operand::isR_M(const Operand &operand)
	{
		return isR_M(operand.type);
	}

	bool Operand::isStr(const Operand &operand)
	{
		return isStr(operand.type);
	}

	const Operand::Register Operand::registerSet[] =
	{
		{VOID,		""},

		{AL,		"al", Encoding::AL},
		{CL,		"cl", Encoding::CL},
		{REG8,		"dl", Encoding::DL},
		{REG8,		"bl", Encoding::BL},
		{REG8,		"ah", Encoding::AH},
		{REG8,		"ch", Encoding::CH},
		{REG8,		"dh", Encoding::DH},
		{REG8,		"bh", Encoding::BH},

		{AX,		"ax", Encoding::AX},
		{CX,		"cx", Encoding::CX},
		{DX,		"dx", Encoding::DX},
		{REG16,		"bx", Encoding::BX},
		{REG16,		"sp", Encoding::SP},
		{REG16,		"bp", Encoding::BP},
		{REG16,		"si", Encoding::SI},
		{REG16,		"di", Encoding::DI},

		{EAX,		"eax", Encoding::EAX},
		{ECX,		"ecx", Encoding::ECX},
		{REG32,		"edx", Encoding::EDX},
		{REG32,		"ebx", Encoding::EBX},
		{REG32,		"esp", Encoding::ESP},
		{REG32,		"ebp", Encoding::EBP},
		{REG32,		"esi", Encoding::ESI},
		{REG32,		"edi", Encoding::EDI},

		{ST0,		"st",  Encoding::ST0},
		{ST0,		"st0", Encoding::ST0},
		{FPUREG,	"st1", Encoding::ST1},
		{FPUREG,	"st2", Encoding::ST2},
		{FPUREG,	"st3", Encoding::ST3},
		{FPUREG,	"st4", Encoding::ST4},
		{FPUREG,	"st5", Encoding::ST5},
		{FPUREG,	"st6", Encoding::ST6},
		{FPUREG,	"st7", Encoding::ST7},

		{MMREG,		"mm0", Encoding::MM0},
		{MMREG,		"mm1", Encoding::MM1},
		{MMREG,		"mm2", Encoding::MM2},
		{MMREG,		"mm3", Encoding::MM3},
		{MMREG,		"mm4", Encoding::MM4},
		{MMREG,		"mm5", Encoding::MM5},
		{MMREG,		"mm6", Encoding::MM6},
		{MMREG,		"mm7", Encoding::MM7},

		{XMMREG,	"xmm0", Encoding::XMM0},
		{XMMREG,	"xmm1", Encoding::XMM1},
		{XMMREG,	"xmm2", Encoding::XMM2},
		{XMMREG,	"xmm3", Encoding::XMM3},
		{XMMREG,	"xmm4", Encoding::XMM4},
		{XMMREG,	"xmm5", Encoding::XMM5},
		{XMMREG,	"xmm6", Encoding::XMM6},
		{XMMREG,	"xmm7", Encoding::XMM7}
	};

	OperandREG Operand::scanReg(const char *string)
	{
		if(string)
		{
			for(int i = 0; i < sizeof(registerSet) / sizeof(Operand::Register); i++)
			{
				if(stricmp(string, registerSet[i].notation) == 0)
				{
					OperandREG reg;

					reg.type = registerSet[i].type;
					reg.reg = registerSet[i].reg;

					return reg;
				}
			}
		}

		return Operand(UNKNOWN);
	}

	const char *Operand::regName() const
	{
		for(int i = 0; i < sizeof(registerSet) / sizeof(Operand::Register); i++)
		{
			if(reg == registerSet[i].reg)
			{
				if(isSubtypeOf(MEM) && Operand::isSubtypeOf(registerSet[i].type, REG32) ||
				   Operand::isSubtypeOf(registerSet[i].type, type) && reg == registerSet[i].reg)
				{
					return registerSet[i].notation;
				}
			}
		}

		throw INTERNAL_ERROR;
	}

	const char *Operand::indexName() const
	{
		for(int i = 0; i < sizeof(registerSet) / sizeof(Operand::Register); i++)
		{
			if(indexReg == registerSet[i].reg && Operand::isSubtypeOf(registerSet[i].type, REG32))
			{
				return registerSet[i].notation;
			}
		}

		throw INTERNAL_ERROR;
	}

	const Operand::Register Operand::syntaxSet[] =
	{
		{VOID,		""},

		{ONE,		"1"},
		{IMM,		"imm"},
		{IMM8,		"imm8"},
		{IMM16,		"imm16"},
		{IMM32,		"imm32"},

		{AL,		"AL"},
		{AX,		"AX"},
		{EAX,		"EAX"},
		{DX,		"DX"},
		{CL,		"CL"},
		{CX,		"CX"},
		{ECX,		"ECX"},
		{ST0,		"ST0"},

		{REG8,		"reg8"},
		{REG16,		"reg16"},
		{REG32,		"reg32"},
		{FPUREG,	"fpureg"},
		{MMREG,		"mmreg"},
		{XMMREG,	"xmmreg"},

		{MEM,		"mem"},
		{MEM8,		"mem8"},
		{MEM16,		"mem16"},
		{MEM32,		"mem32"},
		{MEM64,		"mem64"},
		{MEM128,	"mem128"},

		{R_M8,		"r/m8"},
		{R_M16,		"r/m16"},
		{R_M32,		"r/m32"},
		{R_M64,		"r/m64"},
		{R_M128,	"r/m128"},

		{XMM32,		"xmm32"},
		{XMM64,		"xmm64"},

		{STR,		"str"}
	};

	Operand::Type Operand::scanSyntax(const char *string)
	{
		if(string)
		{
			for(int i = 0; i < sizeof(syntaxSet) / sizeof(Operand::Register); i++)
			{
				if(stricmp(string, syntaxSet[i].notation) == 0)
				{
					return syntaxSet[i].type;
				}
			}
		}

		return UNKNOWN;
	}
}
