/******************************************************************************
* MarchCube.c - An implementation of the marching cube algorithm.	      *
*									      *
*                                                                             *
*                     7            K           6                              *
*                      ***********************                                *
*                    * +                   * *                                *
*                L *   +                 *   *              Vertices 0 - 7    *
*                *     +     I         * J   *              Edges    A - L    *
*            4 *********************** 5     *                                *
*              *       +             *       *                                *
*              *       +             *       * G                              *
*              *       + H           *       *                                *
*              *       +             *       *                                *
*              *       +             * F     *                                *
*            E *       +       C     *       *                                *
*              *       ++++++++++++++*+++++++* 2                              *
*              *   D + 3             *     *                                  *
*              *   +                 *   * B                                  *
*              * +                   * *                                      *
*              ***********************                                        *
*             0           A           1                                       *
*                                                                             *
*******************************************************************************
* Written by Gershon Elber, Dec. 92.					      *
******************************************************************************/

#ifndef MARCH_CUBE_H
#define MARCH_CUBE_H

#include <irit_sm.h>
#include <triv_loc.h>

#define MC_VRTX_0 0
#define MC_VRTX_1 1
#define MC_VRTX_2 2
#define MC_VRTX_3 3
#define MC_VRTX_4 4
#define MC_VRTX_5 5
#define MC_VRTX_6 6
#define MC_VRTX_7 7
#define MC_VRTX_NONE 999

#define MC_EDGE_A 0
#define MC_EDGE_B 1
#define MC_EDGE_C 2
#define MC_EDGE_D 3
#define MC_EDGE_E 4
#define MC_EDGE_F 5
#define MC_EDGE_G 6
#define MC_EDGE_H 7
#define MC_EDGE_I 8
#define MC_EDGE_J 9
#define MC_EDGE_K 10
#define MC_EDGE_L 11
#define MC_EDGE_NONE 999

#define MC_COMP_V_POS(X, Y, Z, Pos) { \
			CCS -> _VrtxPos[Pos][0] = CCS -> Vrtx0Lctn[0] + X; \
			CCS -> _VrtxPos[Pos][1] = CCS -> Vrtx0Lctn[1] + Y; \
			CCS -> _VrtxPos[Pos][2] = CCS -> Vrtx0Lctn[2] + Z; }


typedef struct MCCubeCornerScalarStruct {
    CagdPType Vrtx0Lctn;			  /* Lowest corner position. */
    CagdPType CubeDim; 				    /* Width, Depth, Height. */
    RealType Corners[8];			/* Scalar values of corners. */
    RealType GradientX[8];		    /* Optional gradient at corners. */
    RealType GradientY[8];
    RealType GradientZ[8];
    CagdBType HasGradient;		       /* True if Gradient? are set. */

    /* Used internally. */
    CagdBType _Intersect;
    RealType _Vertices[8];
    CagdPType _VrtxPos[8];
    CagdPType _InterNrml[12];
    CagdPType _InterPos[12];
    int _InterHighV[12];
} MCCubeCornerScalarStruct;

typedef struct MCVertexStruct {
    struct MCVertexStruct *Pnext;		        /* To next in chain. */
    CagdPType V;			       /* Holds X, Y, Z coordinates. */
} MCVertexStruct;

typedef struct MCEdgeStruct {
    struct MCEdgeStruct *Pnext;			        /* To next in chain. */
    CagdPType V[2];
    CagdPType N[2];
} MCEdgeStruct;

typedef struct MCPolygonStruct {
    struct MCPolygonStruct *Pnext;		        /* To next in chain. */
    int NumOfVertices;
    CagdPType V[13];
    CagdPType N[13];
} MCPolygonStruct;

MCPolygonStruct *MCThresholdCube(MCCubeCornerScalarStruct *CCS,
				 RealType Threshold);

#endif /*  MARCH_CUBE_H */
