;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


;; (herald bell-lapadula)

;; You must first have loaded the file $IMPS/theories/exercises/fun-thm-security.t: 

;; (*require nil '(theories exercises/fun-thm-security) imps-implementation-env)

;; Suppose now that we want to introduce a particular notion of secure%state,
;; and thus also to commit ourselves to a particular notion of state.  For this
;; version, we will suppose that there is a fixed function that associates a
;; level with each subject and object.  The main security theorem asserts:
 
;; (def-theorem bell_lapadula-security 
;;   "forall(s:state, accessible(s) implies secure%state(s))"
;;   (theory action_records))

;; That is, we're using the traditional Bell-LaPadula idea that a secure system
;; is one in which there is a notion of secure state, and moreover every state
;; that the system can actually enter is a secure state in that sense.  

;; We will actually introduce the state machine apparatus only quite late in
;; this file, by means of the form

;; (def-theory-ensemble-instances
;;   det-state-machines-with-start
;;   (target-theories action_records)
;;   (sorts (state state) (action action))
;;   (multiples 1)
;;   (theory-interpretation-check using-simplification)
;;   (constants
;;    (next next)
;;    (tr "lambda(s:state, a:action, s_1:state, next(s,a)=s_1)")
;;    (initial "lambda(f:state, f=s_init)")
;;    (s_init s_init)
;;    (accepting "lambda(f:state, falsehood)")))

;; Ignoring the difference between a theory ensemble and an ordinary theory,
;; which is irrelevant in this case, what this form does is to declare that we
;; will regard the theory action_records as an instance of the general theory
;; of deterministic state machines with a distinguished start state.  It says
;; that the transition function "next" of the general theory of deterministic
;; state machines will be associated with the action_records function that
;; happens to be called "next", etc.  It is not important to know the details of
;; the general state machine theories.

;; The state is a FUNCTION.  Given two arguments, namely a subject and an
;; argument, it returns a set of accesses.  Thus, it represents the familiar
;; access control matrix (regarding a matrix as a two-place function).  A state
;; is secure if for every subject/object pair, the simple security and star
;; property hold of the access set.  In the initial state, the access set is
;; null.

;; The theory concerns two basic types.  First are the "users" and "objects"
;; (e.g. files) of the system, which we call jointly "entities."  It is not
;; assumed that the users and objects are disjoint, although in many further
;; applications of the theory they would be.  Next come the security levels
;; themselves, about which we assume that they are equipped with a partial
;; order leq.  level%of is a total function mapping entities to levels.

(def-language entities-and-levels-language
  (base-types entity level)
  (sorts (object entity)
	 (user   entity))
  (constants
   (level%of (entity level))
   (leq (level level prop))))

(def-theory entities-and-levels
  (language entities-and-levels-language)
  (axioms
   (level%of-total
    "total_q{level%of,[entity,level]}"
    d-r-convergence)
   (leq-reflexive
    "forall(l_0:level, leq(l_0, l_0))"
    rewrite)
   (leq-antisymmetric
    "forall(l_0,l_1:level, leq(l_0, l_1) and leq(l_1, l_0) implies l_0=l_1)"
    ())
   (leq-transitive
    "forall(l_0,l_1,l_2:level, leq(l_0, l_1) and leq(l_1, l_2) implies leq(l_0, l_2))"
    ())))

;; The next form defines a theory ACCESS_RECORDS that extends
;; entities-and-levels.  It contains a new base type ACCESS.  An access is a
;; record structure with a read field and a write field (each either true or
;; false).  The selectors will be access_read and access_write.  The usual
;; data-type no-junk and no-confusion axioms are included, as well as rewrite
;; rules access_read(make_access(p,q))=p and access_write(make_access(p,q))=q.
;; This is not yet available through a def-form.

(define access_records
  (make-record-theory-with-sortnames
   (name->theory 'entities-and-levels)
   'access
   '((read "prop")
     (write "prop"))))

(set (current-theory) access_records)
(def-imported-rewrite-rules access_records
  (source-theories family-indicators))


;; Here are some definitions that introduce an initial state, and a subtype of
;; consisting of the possible states.  These states are really functions: When
;; given a user and an object, a state returns an access.  The only condition
;; for a function to be in the subtype is that is should be total (for users
;; and objects).

(def-constant empty%access
  "make_access(falsehood,falsehood)"
  (theory access_records))

;; The initial state of our machine contains no access rights for any
;; user-object pair.    

(def-constant s_init
  "lambda(u:user,o:object, empty%access)"
  (theory access_records))

;; We note that this is a total function:  

(def-theorem s_init-is-total
  "total_q{s_init,[user,object,access]}"
  (theory access_records)
  (usages d-r-convergence)
  ;; Unfold the definition and simplify-insistently 
  (proof ((unfold-single-defined-constant (0) s_init)
	  simplify-insistently)))

;; Now we define the states to be those functions (of this sort) that are
;; total.  Since in the Imps logic, every sort must be non-empty, we supply a
;; witness, namely s_init, which we have just shown to meet this condition.  

(def-atomic-sort state
  "lambda(f:[user,object,access], total_q{f,[user,object,access]})"
  (theory access_records)
  (witness "s_init")
  (usages rewrite))

;; For future convenience we install the fact that s_init is a state as a
;; rewrite.  To prove it, use the macetes available on the macete menu: first
;; state-defining-axiom_access_records, then s_init-is-total.  

(def-theorem s_init-is-state
  "#(s_init,state);"
  (theory access_records)
  (usages rewrite)
  (proof ((apply-macete-with-minor-premises state-defining-axiom_access_records)
	  simplify
	  (apply-macete-with-minor-premises s_init-is-total))))

;; We next establish the duality of read and write.  We do this by supplying a
;; theory interpretation: it interchanges read and write, and inverts the sense
;; of the partial ordering.  It is very useful, as we can prove a theorem about
;; read, and then get the dual theorem about write "for free", using the
;; interpretation.

;; Four obligations must be proved to justify the translation (i.e.  to
;; establish that it is a theory-interpretation).  They are not interesting,
;; and are easy to prove.  They are contained in the file
;; $IMPS/theories/state-machines/bl-exercise-obligations.t
;; The keyword FORCE below tells Imps not to bother about them.   It should be
;; used only when the obligations have been proved and stored in a file.  

(def-translation access_records-symmetry
  force 
  (source access_records)
  (target access_records)
  (sort-pairs (state state))
  (constant-pairs
   (leq					; Invert the ordering 
    "lambda(l_0,l_1:level, leq(l_1,l_0))")		
   (make_access				; Interchange read and write slots 
    "lambda(write,read:prop, 
	make_access(read, write))")
   (access_read access_write)		; map read to write 
   (access_write access_read))		; map write to read 
  (theory-interpretation-check using-simplification))

; SIMPLY%SECURE contains the real content of the notion of secure state in this
; exercise.  A state will be defined to be secure if it has both this and the
; dual property.  Its content is just that if you can read the file, then your
; level dominates its level.   

(def-constant simply%secure
  "lambda(f:state, 
	forall(u:user,o:object, 
	   access_read(f(u,o)) implies leq(level%of(o),level%of(u))))"
  (theory access_records))

;; Here follow two auxiliary definitions, and then the definitions of two of
;; the four state changing operations.  There will be four in all, with the
;; other two being the duals of these two.  With%read%access updates
;; access_read in an access to be true, and without%read%access makes it
;; false.       

(def-constant with%read%access
  "lambda(a:access, make_access(truth,access_write(a)))"
  (theory access_records))

(def-constant without%read%access
  "lambda(a:access, make_access(falsehood,access_write(a)))"
  (theory access_records))

;; State-changing operation to get read access to an object:

(def-constant get%read%access
 "lambda(s:state, u:user, o:object, 
	if(leq(level%of(o), level%of(u)),
	   lambda(u_0:user, o_0:object,
	     if(u=u_0 and o=o_0, with%read%access(s(u,o)), s(u_0,o_0))),
	   s))"
  (theory access_records))

;; State-changing operation to relinquish read access to an object:

(def-constant del%read%access
  "lambda(s:state, u_0:user, o_0:object, 
	lambda(u:user, o:object, if(u=u_0 and o=o_0, without%read%access(s(u,o)), s(u,o))))"
  (theory access_records))

;; Now please load six slightly boring lemmas from an auxiliary file.  They are
;; not hard to prove, but they distract from the main line of the exercise.
;; You may look at them in $IMPS/theories/exercises/aux-bell-lapadula.t if
;; you're interested.  

(load '(theories exercises/aux-bell-lapadula) imps-implementation-env)

;; Here is a series of "security relevant lemmas".  The first asserts that
;; write is irrelevant to simple security.  To prove it, use the command
;; unfold-defined-constants, and then simplify.

(def-theorem simple%security-depends-on-read
  "forall(s_0,s_1:state, 
	forall(u:user, o:object, access_read(s_0(u,o))=access_read(s_1(u,o)))
       implies
	simply%secure(s_0) iff simply%secure(s_1))"
  (theory access_records)
  (usages )
  (proof ((unfold-single-defined-constant-globally simply%secure)
	  (prove-by-logic-and-simplification 0)))) 

;; The next asserts that get%read doesn't affect write.  To prove it,
;; unfold-defined-constants twice, do one direct inference, and then use
;; case-split-on-conditionals twice.

(def-theorem get%read-leaves-write-unchanged
 "forall(s:state,u,u_0:user, o,o_0:object,
	 access_write(get%read%access(s,u_0,o_0)(u,o))=access_write(s(u,o)))"
 (theory access_records)
 (usages )
 (proof ((unfold-single-defined-constant-globally get%read%access)
	 (raise-conditional (0))
	 (unfold-single-defined-constant-globally with%read%access)
	 (raise-conditional (0))
	 (prove-by-logic-and-simplification 1)))) 
