;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald LANGUAGES)


; A language determines a vocabulary for a many-sorted simple type theory in
; which functions may be undefined.  (Predicators may not be.) 

; Languages are divided into two kinds, namely basic and compound languages.
; As we will see below, basic languages are also divided into two kinds, namely
; self-extending and non-self-extending (vanilla).
;
; A basic language is implemented by an object containing a set of constants;
; its base-sorts are all the base-sorts in the sortings of its constants.
; It is assumed that no two basic languages will ever share a single formal
; symbol, though they may certainly contain different formal symbols with the
; same name.
; 
; A compound language is determined by a set of basic languages, together with
; a set of named sorts which may contains symbols not used in the sortings of
; its constants.  Its expressions may use variables involving these sort
; symbols.  A compound language is implemented by a structure.

(define-predicate BASIC-LANGUAGE?)

; The language constants of a language are all those in its vocabulary.  

(define-operation (LANGUAGE-CONSTANTS l))

; Every language has some set of sort symbols.  

(define SORT-SYMBOLS base-sorts)

(define-settable-operation (LANGUAGE-DEFAULT-SORTINGS language))

; A language will keep track of the sexp used to define it:
;

(define-settable-operation (language-defining-form l))


(define-structure-type COMPOUND-LANGUAGE
  basic-languages
  base-sorts
  constants
  default-sorting-alist
  resolver
  extra-resolver					;that part of the resolver that is
							;not needed for constants, just
							;variables 
  name
  table-hash
  defining-form 
  

  (((two-d-table-hash self)
    (let ((hash (compound-language-table-hash self)))
      (if (uncomputed? hash)
	  (let ((hash (descriptor-hash self)))
	    (set (compound-language-table-hash self) hash)
	    hash)
	  hash)))
   ((name self)
    (compound-language-name self))
   ((print self port)
    (format port "#{IMPS-compound-language~_~S:~_~S}"
	    (object-hash self)
	    (name self)))
   ((base-sorts self)
    (compound-language-base-sorts self))
   ((language-constants self)
    (if (not (compound-language-constants self))
	(let ((c (collect-set
		  language-constants
		  (compound-language-basic-languages self))))
	  (set (compound-language-constants self) c)
	  c)
	(let ((self-extending-languages (compound-language-self-extending-languages self)))
	  (if self-extending-languages
	      (let ((c (set-union
			(collect-set language-constants self-extending-languages)
			(compound-language-constants self))))
		(set (compound-language-constants self) c)
		c)
	      (compound-language-constants self)))))
;;;   ((language-constants self)
;;;    (or (compound-language-constants self)
;;;	(let ((c (collect-set
;;;		  language-constants
;;;		  (compound-language-basic-languages self))))
;;;	  (set (compound-language-constants self) c)
;;;	  c)))
   ((sort-resolver? self) '#t)
   ((sort-names-resolved self)
    (sort-names-resolved (compound-language-resolver self)))
   ((sorts-resolved self)
    (sorts-resolved (compound-language-resolver self)))
   ((name-sort-alist self)
    (name-sort-alist (compound-language-resolver self)))
   ((name->sort self symbol)
    (name->sort (compound-language-resolver self) symbol))
   
   ((language-numerical-type->sorting self num-type)
    (any (lambda (basic) (language-numerical-type->sorting basic num-type))
	 (compound-language-basic-languages self)))

   ((language-sorting->numerical-type self sort)
    (any (lambda (basic) (language-sorting->numerical-type basic sort))
	 (compound-language-basic-languages self)))
   
   ((language-default-sortings self) (compound-language-default-sorting-alist self))
   (((setter language-default-sortings) self new-value)
    (set (compound-language-default-sorting-alist self) new-value))
   ((language-defining-form self)
    (compound-language-defining-form self))
   (((setter language-defining-form) self new-value)
    (set (compound-language-defining-form self) new-value))))

(block
  (set (compound-language-constants (stype-master compound-language-stype)) nil)
  (set (compound-language-name (stype-master compound-language-stype)) nil)
  (set (compound-language-table-hash (stype-master compound-language-stype))
       (uncomputed))
  (set (compound-language-default-sorting-alist (stype-master compound-language-stype))
       nil))

; A LANGUAGE is either basic or compound.  

(define (LANGUAGE? l)
  (or (basic-language? l)
      (compound-language? l)))

; The basic languages of a language are all those it contains.  These are the
; language itself, if it is basic, or all its included-languages, if it is
; compound.  

(define (BASIC-LANGUAGES language)
  (if (basic-language? language)
      (list language)
      (compound-language-basic-languages language)))

; L1 is a sublanguage of L2 if its basic languages and named sorts are among
; those of L2.

(define (SUB-LANGUAGE? sub-lang lang)
  (and (subset? (basic-languages sub-lang)
		(basic-languages lang))
       (subset? (sorts-resolved sub-lang)
		(sorts-resolved lang))))

; To find a constant in a compound language, we simply look through all of its
; basic languages.  Each basic language handles this operation by referring to
; its formal symbol chart (see innards-for-languages). 

(define-operation (FIND-CONSTANT language name)
  (iterate iter ((sub (basic-languages language)))
    (and sub
	 (or (find-constant (car sub) name)
	     (iter (cdr sub))))))

(define (FIND-CONSTANT-WITH-WARNING language constant-name)
  (let ((constant (find-constant language constant-name)))
    (if (null? constant)
	(let ((real-names (language-find-constants language constant-name)))
	  (if (= (length real-names) 1)
	      (car real-names)
	      (if real-names
		  (imps-error "FIND-CONSTANT-WITH-WARNING: ~A is actually an overloaded symbol name. Use one of the constant names ~A:" constant-name real-names)
		  (imps-error "FIND-CONSTANT-WITH-WARNING: ~A is not a constant in the language ~A." constant-name language))))
	constant)))


; The home language of a compound expression is the smallest language that
; contains all of its constants and the sort symbols of all of its variables.
; The language itself is retrieved by a call to compound-language (see below). 

(define (COMPUTE-HOME-LANGUAGE components)
  (let* ((const-homes
	  (set-map home-language (collect-set constants components)))
	 (new-var-sorts
	  (set-difference
	   (collect-set 
	    collect-named-sorts 
	    (map expression-sorting (collect-set variables components)))
	   (collect-set sorts-resolved const-homes)))
	 (resolver 
	  (iterate loop ((resolver null-sort-resolver) (sorts new-var-sorts))
	    (if (null? sorts)
		resolver
		(loop (extend-sort-resolver resolver (name (car sorts)) (car sorts))
		      (cdr sorts))))))
    (or (every? language? const-homes)
	(imps-error "COMPUTE-HOME-LANGUAGE: homeless constant in ~S" components))
    (if (and const-homes
	     (null? (cdr const-homes))
	     (null? new-var-sorts))
	(car const-homes)	  
	(compound-language const-homes resolver))))

;;; Commented out by WMF Thu Nov 15 14:26:54 EST 1990
;;;
;;;(define (COMPUTE-HOME-LANGUAGE components)
;;;  (let* ((const-homes
;;;	  (set-map home-language (collect-set constants components)))
;;;	 (new-var-sorts
;;;	  (set-difference
;;;	   (collect-set collect-named-sorts (collect-set variables components))
;;;	   (collect-set sorts-resolved const-homes))))
;;;
;;;    (or (every? language? const-homes)
;;;	(imps-error "COMPUTE-HOME-LANGUAGE: homeless constant in ~S" components))
;;;    (if (and const-homes
;;;	     (null? (cdr const-homes))
;;;	     (null? new-var-sorts))
;;;	(car const-homes)	  
;;;	(compound-language const-homes new-var-sorts))))

; CONTAINS-EXPRESSION? holds of a language and an expression if the latter is
; in the former.

(define (CONTAINS-EXPRESSION? language expression)
  (let ((home (home-language expression)))
    (if (language? home)
	(sub-language? home language)
	(and
	 (every-set-element?
	  (lambda (sort) (contains-sort? language sort))
	  (set-map expression-sorting (variables expression)))
	 (let ((homes (set-map home-language (constants expression))))
	   (every?
	    (lambda (home)
	      (sub-language? home language))
	    homes))))))

(define (LANGUAGE-CONTAINS-SORTING? language sorting)
  (subset? (base-sorts sorting) (base-sorts language)))

; COMPOUND-EXPRESSION either retrieves the expression with the given
; constructor and component list, or constructs a new expression, computing its
; home language and entering it into the table.

(define (COMPOUND-EXPRESSION constructor components)
  (apply constructor components))

; NAME->LANGUAGE is a settable operation that keeps an alist to retrieve
; languages given their names. 

(define NAME->LANGUAGE
  (settable-symbol-alist language? 'warn-on-redefining))


; The basic way to make basic languages.  The symbol form becomes the name of
; the language unless it is nil.  The new-constants, which must not yet have
; homes, become its constants.  

(define (MAKE-BASIC-LANGUAGE symbol-form new-constants resolver)
  (if (and (not (symbol? symbol-form))
	   (not (eq? symbol-form nil)))
      (imps-error "MAKE-BASIC-LANGUAGE: unsuitable name ~A" symbol-form))
  (if
   (and (null? new-constants)
	(subset? (sort-names-resolved resolver) (list ind prop)))
   the-null-language
   (let* ((constant-chart
	   (install-constants-in-chart new-constants))
	  (default-sorting-alist '())
	  (table-hash (descriptor-hash constant-chart))
	  (defining-form '()))
     (let ((obj
	    (object nil
	      ((basic-language? self) '#t)
	      ((name self) symbol-form)
	      ((print self port) 
	       (format port "#{IMPS-basic-language~_~S:~_~S}"
		       (object-hash self)
		       symbol-form))
	      ((two-d-table-hash self) table-hash)
	      ((base-sorts self) (base-sorts resolver))
	      ((language-constants self)
	       (glean-formal-symbols-from-chart constant-chart))

	      ((find-constant self the-name)
	       (retrieve-formal-symbol constant-chart the-name))
	      ((name->sort self symbol)
	       (name->sort resolver symbol))
	      ((sort-resolver? self) '#t)
	      ((sort-names-resolved self)
	       (sort-names-resolved resolver))
	      ((sorts-resolved self)
	       (sorts-resolved resolver))
	      ((name-sort-alist self)
	       (name-sort-alist resolver))
 
	      ((language-default-sortings self) default-sorting-alist)
	      (((setter language-default-sortings) self new-value)
	       (set default-sorting-alist new-value))

	      ((language-defining-form self) defining-form)
	      (((setter language-defining-form) self new-value)
	       (set defining-form new-value)))))
       (walk
	(lambda (constant)
	  (if (home-language constant)
	      (imps-error "MAKE-LANGUAGE:  Constant ~A is not new" constant)
	      (set (expression-home constant) obj)))
	new-constants)
       (set (find-language (list obj) (sorts-resolved resolver)) obj)
       (if (symbol? symbol-form)
	   (set (name->language symbol-form) obj))
       obj))))

; A self-extending language is one that incorporates new formal symbols when
; encountered.  For instance, we might want to be able to use a kind of numeral
; in a language.  There will normally be a potential infinity of possible
; numerals, so we simply create formal symbols for the ones we actually
; encounter.  To do this we use an alist with predicate-sorting pairs.  On
; finding a symbol name, we get its intended type by running down the alist,
; testing with the predicates.  The intended sorting is then the cdr of the
; selected pair.  Thus for instance we might use ((fixnum? 'nn)(rational? 'qq))
; to force a new fixnum into sort 'nn and any other kind of rational into 'qq.  

(define-predicate SELF-EXTENDING?) 

(define-operation (predicate-sorting-alist->sexp language)
  '#f)
(define-operation (language-predicate-sorting-alist language)
  '#f)
(define-operation (language-numerical-type->sorting language num-type))
(define-operation (language-sorting->numerical-type language sorting)
  '#f)

(define (MAKE-SELF-EXTENDING-BASIC-LANGUAGE symbol-form new-constants
					    resolver predicate-sorting-alist)
  (if (and (not (symbol? symbol-form))
	   (not (eq? symbol-form nil)))
      (imps-error "make-self-extending-basic-language: unsuitable name ~A" symbol-form))

  (walk
   (lambda (pair)
     (let ((sort (cdr pair)))
       (if (base-sort? sort)
	   (set (base-sort-numerical? sort) (car pair))
	   (set (higher-sort-numerical? sort) (car pair)))))
   predicate-sorting-alist)
 
  (let* ((constant-chart
	  (install-constants-in-chart new-constants))
	 (default-sorting-alist '())
	 (table-hash (descriptor-hash constant-chart))
	 (defining-form '())) 
    (let ((obj
	   (object nil
	     ((basic-language? self) '#t)
	     ((self-extending? self) '#t)
	     ((name self) symbol-form)
	     ((print self port) 
	      (format port "#{IMPS-basic-language~_~S:~_~S}"
		      (object-hash self)
		      symbol-form))
	     ((two-d-table-hash self) table-hash)
	     ((base-sorts self) (base-sorts resolver))
	     ((language-constants self)
	      (glean-formal-symbols-from-chart constant-chart))

	     ((find-constant self the-name)
	      (let ((sorting
		     (self-extending-language-const-name->sorting
		      the-name predicate-sorting-alist)))
		(retrieve-or-add-formal-symbol self constant-chart the-name sorting)))
	     
	     ((name->sort self symbol)
	      (name->sort resolver symbol))
	     ((sort-resolver? self) '#t)
	     ((sort-names-resolved self)
	      (sort-names-resolved resolver))
	     ((sorts-resolved self)
	      (sorts-resolved resolver))
	     ((name-sort-alist self)
	      (name-sort-alist resolver))

	     ((language-numerical-type->sorting self num-type)
	      (let ((pair (assq num-type predicate-sorting-alist)))
		(and (pair? pair)
		     (cdr pair))))

	     ((language-sorting->numerical-type self sort)
	      (let ((pair (ass-lq sort predicate-sorting-alist)))
		(and (pair? pair)
		     (car pair))))

	     ((language-predicate-sorting-alist self)
	      predicate-sorting-alist)
	     
	     ((predicate-sorting-alist->sexp self) ;used when printing a 
	      (map			;definition for the   
	       (lambda (pair)		;language             
		 (cons
		  (procedure-name (car pair))
		  (cdr pair)))		   
	       predicate-sorting-alist))
	     ((language-default-sortings self) default-sorting-alist)
	     (((setter language-default-sortings) self new-value)
	      (set default-sorting-alist new-value))
	     
	      ((language-defining-form self) defining-form)
	      (((setter language-defining-form) self new-value)
	       (set defining-form new-value)))))
      (walk
       (lambda (constant)
	 (if (home-language constant)
	     (imps-error "MAKE-LANGUAGE:  Constant ~A is not new" constant)
	     (set (expression-home constant) obj)))
       new-constants)
      (set (find-language (list obj) (sorts-resolved resolver)) obj)
      (if (symbol? symbol-form)
	  (set (name->language symbol-form) obj))
      obj)))

(define (SELF-EXTENDING-LANGUAGE-CONST-NAME->SORTING the-name the-alist)
  (iterate iter ((the-alist the-alist))
    (if (null? the-alist)
	'#f
	(let ((first (caar the-alist)))
	  (cond ((and (numerical-type? first)
		      ((numerical-type-recognizer first) the-name))
		 (cdar the-alist))
		((and (procedure? first)
		      (first the-name))
		 (cdar the-alist))
		(else (iter (cdr the-alist))))))))

(define (COMPOUND-LANGUAGE-SELF-EXTENDING-LANGUAGES language)
  (or (compound-language? language)
      (imps-error "COMPOUND-LANGUAGE-SELF-EXTENDING-LANGUAGES: ~A ~S."
		  language "is not a compound language"))
  (set-separate self-extending? (compound-language-basic-languages language)))


; To make a compound language that includes a list of languages, together with
; some extra sorts, we compute the set of basic languages contained and the set
; of all sort symbols.  We consult FIND-LANGUAGE to check if such a beast
; exists, and if not create it.  
;; 
(define (COMPOUND-LANGUAGE languages extra-resolver . symbol-form)
  (if (and symbol-form (not (symbol? (car symbol-form))))
      (imps-error "COMPOUND-LANGUAGE: unsuitable name ~A" symbol-form))
  (let ((consts (collect-set constants languages))
	(resolvers (if (sort-resolver? extra-resolver)
		       (cons extra-resolver languages)
		       languages)))
    (cond 
     ((compound-language-constant-name-conflict? consts)
      => (lambda (the-name)
	   (imps-error "COMPOUND-LANGUAGE: overloaded name ~A" the-name)))
     ((compound-language-sort-name-conflict? resolvers)
      => (lambda (the-name)
	   (imps-error "COMPOUND-LANGUAGE: overloaded sort name ~A" the-name))))
    (let ((resolver (join-sort-resolvers resolvers))
	  (symbol-form (car symbol-form)))
      (cond
       ((find-language languages (sorts-resolved resolver))
	=>
	(lambda (l)
	  (if (and symbol-form
		   (not (eq? (name->language symbol-form)
			     l)))
	      (set (name->language symbol-form) l))
	  l))
       (else 
	(let ((sublangs (collect-set basic-languages languages))
	      (lang (make-compound-language)))
	  (set (compound-language-basic-languages lang) sublangs)
	  (set (compound-language-base-sorts lang) (base-sorts resolver))
	  (set (compound-language-constants lang) consts)	     
	  (set (compound-language-resolver lang) resolver)
	  (set (compound-language-name lang) symbol-form)
	  (set (compound-language-extra-resolver lang) extra-resolver)
	  (set (find-language sublangs (sorts-resolved resolver)) lang)
	  (if symbol-form
	      (set (name->language symbol-form) lang))
	  lang))))))

(lset *COMPOUND-LANGUAGE-CONSTANT-NAME-CONFLICT-table*
      (make-table '*COMPOUND-LANGUAGE-CONSTANT-NAME-CONFLICT-table*))

(define (COMPOUND-LANGUAGE-CONSTANT-NAME-CONFLICT? consts)
  (let ((table *COMPOUND-LANGUAGE-CONSTANT-NAME-CONFLICT-table*))
    (iterate name-conflict? ((consts consts))
      (if (null? consts)
	  '#f
	  (let ((name-1 (name (car consts))))
	    (or (table-entry table name-1)
		(bind (((table-entry table name-1) name-1))
		  (name-conflict? (cdr consts)))))))))


(lset *COMPOUND-LANGUAGE-SORT-NAME-CONFLICT-table*
      (make-table '*COMPOUND-LANGUAGE-SORT-NAME-CONFLICT-table*))

(define (COMPOUND-LANGUAGE-SORT-NAME-CONFLICT? resolvers)
  (let ((table *compound-language-sort-name-conflict-table*))
    (iterate name-conflict? ((resolvers resolvers))
      (if (null? resolvers)
	  '#f
	  (iterate insert ((sorts (sorts-resolved (car resolvers))))
	    (if (null? sorts)
		(name-conflict? (cdr resolvers))
		(let* ((sort-1 (car sorts))
		       (entry (table-entry table (name sort-1))))
		  (or (and (not (eq? entry sort-1))
			   entry)			   
		      (bind (((table-entry table (name sort-1)) sort-1))
			(insert (cdr sorts)))))))))))

; FIND-LANGUAGE is a settable operation that keeps a two-dimensional alist of
; languages by basic languages and named sorts.  

(let ((language-alist (list '*language-alist*)))
  (define FIND-LANGUAGE  
    (operation 
	(lambda (lang-set sort-set)
	  (let* ((sort-set
		  (or sort-set
		      (collect-set sorts-resolved lang-set)))
		 (basics
		  (if (every? basic-language? lang-set)
		      lang-set
		      (collect-set basic-languages lang-set)))
		 (sublist (cdr (ass equal-sets? basics (cdr language-alist)))))
	    (cond ((ass equal-sets? sort-set sublist)
		   => cdr)
		  (else '#f))))
      ((setter self)
       (lambda (lang-set sort-set new-value)
	 (let* ((sort-set
		 (or sort-set
		     (collect-set sorts-resolved lang-set)))
		(basics
		 (if (every? basic-language? lang-set)
		     lang-set
		     (collect-set basic-languages lang-set)))
		(sublist (cdr (ass equal-sets? basics (cdr language-alist)))))
	   (cond ((ass equal-sets? sort-set sublist)
		  =>
		  (lambda (pair)(set (cdr pair) new-value)))
		 (sublist
		  (let ((rem (cdr sublist)))
		    (set (cdr sublist)
			 (cons (cons sort-set new-value)
			       rem))))
		 (else
		  (let ((rem (cdr language-alist)))
		    (set (cdr language-alist)
			 (cons
			  (cons basics
				(list (cons sort-set new-value)))
			  rem))))))))))

  (define (list-languages)
    (collect-set 
     (lambda (sublist)
       (map-set
	cdr	  
	(cdr sublist)))
     (cdr language-alist))))

; The null language is a compound language lacking basic languages and sort
; symbols (other than PROP and IND).  

(define THE-NULL-LANGUAGE
  (compound-language nil nil 'the-null-language))

; To extend a language means to add a set of constants to it.  These constants
; may or may not be new in the sense of not yet belonging to a basic language.
; The result is the smallest language including the given language and
; containing the constants.  If they have a home language to begin with, the
; resulting language will contain anything else in those home languages.
; Symbol-form is an optional name for the resulting language.  The argument
; RESOLVER is assumed to be a sort resolver that covers all the sorts in the
; language and the new constants.   

(define (EXTEND-LANGUAGE language new-consts resolver . symbol-form)
  (cond ((set-separate
	  (lambda (e) (not (home-language e)))
	  new-consts)
	 => (lambda (homeless)
	      (make-basic-language nil homeless resolver))))
  (let ((langs
	 (add-set-element language
			  (set-map home-language new-consts))))
    (apply compound-language langs resolver symbol-form)))

; Given a set of languages, language-union returns the language whose constants
; are the union of theirs, and whose sort symbols are the union of theirs. 

(define (LANGUAGE-UNION languages . symbol-form)
  (if (every? language? languages)
      (apply compound-language
	     languages
	     null-sort-resolver
	     symbol-form)
      (imps-error "LANGUAGE-UNION: ~A is not a list of languages." languages)))


(define (rename-in-basic-language language renamer)
  (let ((self-extending? (self-extending? language))
	(resolver (rename-sort-resolver language renamer))
	(new-constants
	 (map
	  (lambda (c)
	    (let ((s (expression-sorting c))
		  (n (expression-name    c)))
	      (make-formal-symbol constant?
				  (rename-sort s renamer)
				  (renamer n))))
	  (language-constants language))))
    (if self-extending?
	(make-self-extending-basic-language
	  '()
	  new-constants
	  resolver
	  (map (lambda (pair)
		 (cons (car pair)
		       (rename-sort (cdr pair) resolver)))
	       (language-predicate-sorting-alist language)))
	(make-basic-language '() new-constants resolver))))

(define (rename-in-compound-language language fixed renamer)
  (let ((basics
	 (map
	  (lambda (l)
	    (if (any? (lambda (f) (sub-language? l f))
		      fixed)
		l
		(rename-in-basic-language l renamer)))
	  (compound-language-basic-languages language)))
	(extra-resolver (rename-sort-resolver
			 (compound-language-extra-resolver language)
			 renamer)))
    (compound-language basics extra-resolver)))

(define (rename-in-language language fixed renamer)
  (cond ((compound-language? language)
	 (rename-in-compound-language language fixed renamer))
	((sub-language? language fixed) language)
	((basic-language? language)
	 (rename-in-basic-language language renamer))
	(else (imps-error "rename-in-language:  wierdo language ~S." language))))


; language-difference returns the language whose constants are those in lang1
; but not lang2, and whose EXTRA sorts are those in lang1 not occurring in
; lang2.  We cannot infer that a sort symbol used in lang2 will not be present
; in the difference.  

(define (LANGUAGE-DIFFERENCE lang1 lang2 . symbol-form)
  (apply compound-language
	 (set-difference (basic-languages lang1) (basic-languages lang2))
	 (subtract-sort-resolvers lang1 lang2)
	 symbol-form))

; Next a routine for constructing a language from a definition such as:
;  
; (arithmetic
;  (extensible (non-negative-integer? nn)
;	       (rational? qq))
;  (base-types vector point)                     ; This used to be FLOATER-TYPES.
;  (propositional-floater-types super-set)       ; This is now obsolete!
;  (named-sorts (nn ind)
;		(rr ind)
;		(real-fn (rr rr))
;		(vector-length (vector rr)))
;  (constants
;   (0 nn)
;   (1+ (nn nn))
;   (+ (nn nn nn))
;   (* (nn nn nn))
;   (< (nn nn PROP))
;   (q+ (qq qq qq))
;   (q* (qq qq qq))
;   (q< (qq qq PROP))
;   (numerator (qq nn))
;   (denominator (qq nn))
;   (make-qq (nn nn qq))))
;  
; The ``extensible'' form is optional here.  Also permitted is an
; ``embedded-languages'' form such as (embedded-language arithmetic) or
; (embedded-languages integer-arithmetic rational-arithmetic).  Because
; self-extending languages are always basic, and a definition using an
; ``embedded-languages'' form is necessarily compound, these two optional forms
; are not permitted together.  In embedded-languages, either the singular or
; the plural is permitted.   

(define (LANGUAGE-FROM-DEFINITION form)
  (let ((name (car form))
	(embedded (or (assq-val 'embedded-language (cdr form))
		      (assq-val 'embedded-languages (cdr form))))
	(self-extending-alist (assq-val 'extensible (cdr form))))
    (let ((language 
	   (cond
	    ((and embedded self-extending-alist) 
	     (imps-error "LANGUAGE-FROM-DEFINITION: If an embedded language is present, the language can not be extensible."))
	    (embedded
	     (complex-language-from-definition (map (lambda (n)
						      (or (name->language n)
							  (theory-language
							   (name->theory n))))
						    embedded)
					       (cdr form)
					       name))
	    (self-extending-alist
	     (self-extending-language-from-definition
	      self-extending-alist
	      (cdr form)
	      name))
	    (else
	     (basic-language-from-definition (cdr form) name)))))
      (set (language-defining-form language) form)
      language)))

(define (sort-resolver-from-definition embedded-languages form)
  (let ((floaters (cond ((assq 'base-types form) => cdr)
			((assq 'floater-types form) => cdr)
			(else '())))
	(prop-floaters (cond ((assq 'propositional-floater-types form) => cdr)
			     (else '()))))
    (or (null? prop-floaters)
	(imps-warning "sort-resolver-from-definition: ~A ~S~%"
		      "Propositional floating types are now obsolete"
		      prop-floaters))
    (iterate iter ((resolver (join-sort-resolvers 
			      (cons (make-type-resolver floaters prop-floaters)
				    embedded-languages)))
		   (sorting-specs (cond ((assq 'named-sorts form) => cdr)
					((assq 'atomic-sorts form) => cdr)
					((assq 'sorts form) => cdr)
					(else '()))))
	     (if (null? sorting-specs)
		 resolver
		 (iter (sort-resolver-from-definition-process-spec
			resolver
			(car sorting-specs))
		       (cdr sorting-specs))))))
							    
(define (sort-resolver-from-definition-process-spec resolver spec)
  (let* ((sort-name (car spec))
	 (enclosing-sort  (string-or-list->sort resolver (cadr spec)))
	 (new-sort (if (base-sort? enclosing-sort)
		       (build-base-sort
			sort-name
			(sort-category enclosing-sort)
			'#f
			enclosing-sort)
		       (build-higher-sort
			(higher-sort-domains enclosing-sort)
			(higher-sort-range enclosing-sort)
			sort-name
			enclosing-sort))))
    (extend-sort-resolver
     resolver
     sort-name	
     new-sort))) 

(define (MAKE-LANGUAGE-CONSTANTS constant-specs resolver)
  (map
   (lambda (sym-spec)
     (destructure (((name sorting-spec) sym-spec))
       (let ((sorting (string-or-list->sort resolver sorting-spec)))
	 (or (possible-symbol-form? name)
	     (imps-error
	      "MAKE-LANGUAGE-CONSTANTS: formal symbol may not be named ~A" name))
	 (or sorting
	     (imps-error
	      "MAKE-LANGUAGE-CONSTANTS: sort ~A unreadable in language being defined."
	      sorting-spec))
	 (make-formal-symbol constant? sorting name))))
   constant-specs))
	     
(define (complex-language-from-definition embedded-languages form name)
  (let* ((resolver (sort-resolver-from-definition embedded-languages form))
	 (constants (cond ((assq 'constants form)
			   => (lambda (constant-clause)
				(make-language-constants (cdr constant-clause) resolver)))
			  (else '()))))
    (extend-language
     (language-union embedded-languages)	    
     constants
     resolver
     name)))
    
(define (self-extending-language-from-definition self-extending-alist form name)
  (let* ((resolver (sort-resolver-from-definition '() form))
	 (constants (cond ((assq 'constants form)
			   => (lambda (constant-clause)
				(make-language-constants (cdr constant-clause) resolver)))
			  (else '())))
	 (alist (map					;predicate-sorting alist
		 (lambda (pair)
		   (cons (eval (car pair) (the-environment)) ;the predicate 
			 (list->sort resolver (cadr pair)))) ;the sort
		 self-extending-alist)))
    (make-self-extending-basic-language name constants resolver alist)))

(define (basic-language-from-definition form name)
  (let* ((resolver (sort-resolver-from-definition '() form))
	 (constants (cond ((assq 'constants form)
			   => (lambda (constant-clause)
				(make-language-constants (cdr constant-clause) resolver)))
			  (else '()))))
    (make-basic-language name constants resolver)))

(define (MAKE-FORMAL-CONSTANT-IN-NEW-LANGUAGE resolver sorting name)
  (let ((new-const
	 (make-formal-symbol constant? sorting name))
	(language-name (retrieve-unused-name name->language 'new-constant)))
    (make-basic-language language-name (list new-const) resolver)
    new-const))

(define (MAKE-FORMAL-CONSTANTS-IN-NEW-LANGUAGE resolver sorting-list name-list)
  (let ((new-constants (iterate iter ((s-list sorting-list)
				      (n-list name-list)
				      (c-list '()))
			 (if (null? s-list)
			     (reverse c-list)
			     (let ((new-const (make-formal-symbol
					       constant?
					       (car s-list)
					       (car n-list))))
			       (iter (cdr s-list)
				     (cdr n-list)
				     (cons new-const c-list))))))
	(language-name (retrieve-unused-name name->language 'new-constants)))
    (make-basic-language language-name new-constants resolver)
    new-constants))
		   
(define (FIND-BASIC-LANGUAGE language constant)
  (iterate loop ((basics (basic-languages language)))
    (cond ((null? basics)
	   '#f)
	  ((element-of-set? constant (language-constants (car basics)))
	   (car basics))
	  (else
	   (loop (cdr basics))))))


;;; SHRINK-LANGUAGE returns the maximal sublanguage of LANGUAGE not containing
;;; CONSTANTS and NAMED-SORTS.
  
(define (SHRINK-LANGUAGE language constants named-sorts)
  (let ((constants-basic-languages 
	 (map 
	  (lambda (const) (find-basic-language language const))
	  constants))
	(named-sorts-resolver (make-sort-resolver-from-named-sorts named-sorts)))
    (compound-language
     (set-difference (basic-languages language) constants-basic-languages)
     (subtract-sort-resolvers language named-sorts-resolver))))

(define (MAKE-SUBLANGUAGE 
	 the-name superlanguage-name language-names sort-names constant-names)
  (or superlanguage-name
      (imps-error "MAKE-SUBLANGUAGE: The superlanguage name is empty."))
  (or language-names sort-names constant-names
      (imps-error "MAKE-SUBLANGUAGE: There are no language, sort, or constant names given."))
  (let* ((superlanguage
	  (or (name->language superlanguage-name)
	      (theory-language (name->theory superlanguage-name))))
	 (languages 
	  (map 
	   (lambda (l-name)
	     (or (name->language l-name)
		 (theory-language (name->theory l-name))))
	   language-names))
	 (sorts 
	  (map
	   (lambda (name) (name->sort superlanguage name))
	   sort-names))
	 (constants
	  (map
	   (lambda (name) (find-constant superlanguage name))
	   constant-names)))
    (every?
     (lambda (lang)
       (or (sub-language? lang superlanguage)
	   (imps-error "MAKE-SUBLANGUAGE: ~A ~S ~A ~S."
		       "The language" lang "is not a sublanguage of" superlanguage)))
     languages)
    (extend-language 
     (language-union languages)
     constants
     (make-sort-resolver-from-named-sorts sorts)
     the-name)))
