;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald INNARDS-FOR-LANGUAGES)


; Variables are kept in a single global table.  They may be
; retrieved by name and sorting.  

(lset *imps-variable-table* (make-two-d-table '*imps-variable-table*))

; FIND-VARIABLE takes a name and a sorting and retrieves the variable with
; those attributes if it exists.  Otherwise it creates it and stores it in the
; global variable table.  

(define (FIND-VARIABLE name sorting)
  (if (not (symbol? name))
      (imps-error "FIND-VARIABLE: variable-name must be a symbol: ~S" name))
  (if (not (sorting? sorting))
      (imps-error "FIND-VARIABLE: the sorting ~S for variable ~A is not found in current language"
		  sorting name))
  (let ((entry (two-d-table-entry *imps-variable-table* name sorting)))
    (or
     entry      
     (let ((new-var (make-formal-symbol variable? sorting name)))
       (set (two-d-table-entry *imps-variable-table* name sorting)
	    new-var)
       new-var))))

; Return a variable of sorting SORTING, name beginning with the symbol SYM,
; and not in the set of variables AVOID-VARS.  If AVOID-VARS contains a
; variable already named SYM, we recurse, trying the new symbol $SYM. 

(define (NEW-VARIABLE sorting sym avoid-vars)
  (or (symbol? sym)
      (imps-error "NEW-VARIABLE : non-symbol argument ~S." sym))
  (let ((str (symbol->string sym))
	(avoid-strs (map
		     (lambda (v)
		       (if (symbol? (name v))
			   (symbol->string (name v))
			   ""))
		     avoid-vars)))
    (if (not (mem? string-equal? str avoid-strs)) 
	(find-variable sym sorting)
	(let* ((str (string-var-root str))
	       ;;;This was an ordinary "let" statement. But this was clearly wrong. 
	       (avoid-strs (set-separate
			    (lambda (avoid)
			      (substring? str avoid))
			    avoid-strs)))
	  (iterate iter ((i 0))
	    (let ((guess (format nil "~A_$~D" str i)))
	      (if (mem? string-equal? guess avoid-strs)
		  (iter (fx+ i 1))
		  (find-variable (string->symbol guess) sorting))))))))
	       
	       
(define (clip-at-last-$ string)
  (substring string 0 (last-$-pos string)))

(define (last-$-pos string)
  (let ((len (string-length string)))
    (iterate iter ((i 0)
		   (pos '#f))
      (cond ((fx= i len) pos)
	    ((char= #\$ (string-elt string i))
	     (iter (fx+ i 1) i))
	    (else (iter (fx+ i 1) pos))))))

(define (string-var-root str)
  (let ((len (string-length str)))
    (iterate iter ((i (fx- len 1))
		   (digits-too '#t))
      (cond ((fx< i 0) (imps-error "string-var-root : root not found in ~S" str))
	    ((and digits-too (digit? (string-elt str i) 10))
	     (iter (fx- i 1) '#t))
	    ((char= (string-elt str i) #\$)
	     (iter (fx- i 1) '#f))
	    ((char= (string-elt str i) #\_)
	     (iter (fx- i 1) '#f))
	    (else
	     (substring str 0 (fx+ i 1)))))))
	    

(define (undollar-variable var)
  (let ((n (name var)))
    (if (symbol? n)
	(find-variable (string->symbol (clip-at-last-$ (symbol->string n)))
		       (expression-sorting var))
	var)))

; When a number of different new variables of the same kind are needed,
; new-variableS is appropriate.  

(define (NEW-VARIABLES sorting sym avoid-vars how-many)
  (iterate iter ((i 0)
		 (new-variables nil)
		 (avoid avoid-vars))
    (if (>= i how-many)
	(reverse! new-variables)
	(let ((new (new-variable sorting
				 (concatenate-symbol sym '_ i)
				 avoid)))
	  (iter (1+ i)
		(cons new new-variables)
		(cons new avoid))))))

; When a list of sorts is given, and a list of different new variables of those
; sorts is needed, SORTS->NEW-VARIABLES is appropriate.

(define (SORTS->NEW-VARIABLES sorting-list sym avoid-vars)
  (iterate iter ((i 0)
		 (sorting-list sorting-list)
		 (new-variables nil)
		 (avoid-vars avoid-vars))
    (if (null? sorting-list)
	(reverse! new-variables)
	(let ((new (new-variable (car sorting-list)
				 (concatenate-symbol sym '_ i)
				 avoid-vars)))
	  (iter (1+ i)
		(cdr sorting-list)
		(cons new new-variables)
		(cons new avoid-vars))))))

(define (SORTS-AND-NAMES->NEW-VARIABLES sorting-list sym-list avoid-vars)
  (iterate iter ((sorting-list sorting-list)
		 (sym-list sym-list)
		 (new-variables nil)
		 (avoid-vars avoid-vars))
    (if (null? sorting-list)
	(reverse! new-variables)
	(let ((new (new-variable (car sorting-list)
				 (car sym-list)
				 avoid-vars)))
	  (iter (cdr sorting-list)
		(cdr sym-list)
		(cons new new-variables)
		(cons new avoid-vars))))))
		   

; When a list of variables is given, and new variables of the corresponding
; sorts are needed, use VAR-LIST->NEW-VARIABLES.  VAR-LIST may have multiple
; occurrences, in which case so will the list of new variables.  

(define (VAR-LIST->NEW-VARIABLES var-list avoid-vars)
  (iterate iter ((var-list var-list)
		 (new-var-list nil))
    (if (null? var-list)
	(reverse! new-var-list)
	(let* ((first (car var-list))
	       (new (new-variable (expression-sorting first) (name first) avoid-vars)))
	  (iter (cdr var-list)
		(cons new new-var-list))))))


(define (MARK-VARIABLE variable sym avoid-vars)
  (new-variable
   (expression-sorting variable)
   (concatenate-symbol (name variable) '_ sym)
   avoid-vars))

(define (MARK-VARIABLES variable-list sym avoid-vars)
  (let ((name-list (map
		    (lambda (var)
		      (concatenate-symbol (name var) '_ sym))
		    variable-list))
	(sorting-list (map expression-sorting variable-list)))
    (sorts-and-names->new-variables sorting-list name-list avoid-vars)))
   

; The innards of a LANGUAGE consists mainly in a formal-symbol-chart.  This is
; a data structure to hold the contants of the language.  They are retrieved by
; name and sorting.

; Given a name and sorting, one wants to be able to retrieve a formal-symbol
; matching them; and correspondingly, one wants to be able to add a formal
; symbol, given it.

(define (FORMAL-SYMBOL-CHART-ASS obj alist)
  (ass
   (lambda (obj key)
     (or (eq? obj key)
	 (and (numerical-object? obj)
	      (numerical-object? key)
	      (numerical-= obj key))))
   obj alist))

(define (MAKE-FORMAL-SYMBOL-CHART)
  (list '*FORMAL-SYMBOL-CHART*))

(define (ADD-FORMAL-SYMBOL-TO-CHART formal-symbol-chart local-name formal-symbol)
  (let ((sub-chart (formal-symbol-chart-ass local-name (cdr formal-symbol-chart))))
    (cond
     ((and sub-chart (memq? formal-symbol sub-chart)))
     (sub-chart
      (set (cdr sub-chart) (cons formal-symbol (cdr sub-chart))))
     (else
      (let ((remainder (cdr formal-symbol-chart)))
	(set (cdr formal-symbol-chart)
	     (cons
	      (list local-name formal-symbol)
	      remainder)))))
    formal-symbol))

(define (RETRIEVE-FORMAL-SYMBOL chart the-name . sorting)
  (let ((sub-chart (formal-symbol-chart-ass the-name (cdr chart))))
    (cond
     ((not sub-chart) '#f)				;no entry
     ((null? (cddr sub-chart))				;if just one entry
      (cadr sub-chart))					;take it
     (else
      (let ((sorting (car sorting)))			;else require match on 
	(iterate search ((candidates (cdr sub-chart)))	;sorting 
	  (if (null? candidates)
	      '#f
	      (let ((candidate (car candidates)))
		(if (sortings-equal? sorting
			    (expression-sorting candidate))
		    candidate
		    (search (cdr candidates)))))))))))

; Walk through a chart and retrieve all the symbols in it.  

(define (GLEAN-FORMAL-SYMBOLS-FROM-CHART chart)
  (iterate iter ((chart (cdr chart))(already nil))
    (if (null? chart)
	already
	(iter (cdr chart)
	      (append (cdar chart) already)))))

; A procedure that retrieves a constant if one is present in the formal symbol
; chart, or creates one and puts it there otherwise.  It is used by
; self-extending languages.  

(define (RETRIEVE-OR-ADD-FORMAL-SYMBOL home chart the-name sorting)
  (cond (sorting
	 (or (retrieve-formal-symbol chart the-name sorting)
	     (let ((new-constant
		    (make-formal-symbol constant? sorting the-name home)))
	       (add-formal-symbol-to-chart chart the-name new-constant)
	       new-constant)))
	((retrieve-formal-symbol chart the-name))
	(else '#f)))	     

; A procedure to walk through a list of constants and add them all to a newly
; constructed chart.  

(define (INSTALL-CONSTANTS-IN-CHART new-symbols)
  (let ((chart (make-formal-symbol-chart)))	     
    (walk
     (lambda (c)
       (add-formal-symbol-to-chart chart (name c) c))
     new-symbols)
    chart))

; The *compound-expression-table* keeps a unique instance for each compound
; ideal expression active.  These are organized by expression constructor and
; components.

(lset *compound-expression-table* (make-two-d-table '*compound-expression-table*))   

(define (RETRIEVE-FROM-COMPOUND-TABLE constructor components)
  (let ((entry-list
	 (two-d-table-entry
	  *compound-expression-table*
	  constructor
	  (hash-combine-exprs components))))
    (iterate search ((candidates entry-list))
      (if (null? candidates)
	  '#f
	  (let ((comp (expression-components (car candidates))))
	    (if (equal? comp components)
		(car candidates)
		(search (cdr candidates))))))))

(define (INSERT-IN-COMPOUND-TABLE constructor components expression)
  (let ((computed-hash (hash-combine-exprs components)))
    ;; (let ((current-entries
    ;;   (two-d-table-entry *compound-expression-table* constructor computed-hash)))
    ;; (if (memq? expression current-entries)
    ;; 	  (imps-error "INSERT-IN-COMPOUND-TABLE failed, consult Josh.~&~S"
    ;; 		      (the-environment)))
    ;;       (set (expression-alpha-root expression)
    ;; 	   (find-alpha-root-in-list expression current-entries))
    ;;       (or (expression? (expression-alpha-root expression))
    ;; 	  (imps-error "INSERT-IN-COMPOUND-TABLE failed, consult Josh.~&~S"
    ;; 		      (the-environment)))) 
    (push
     (two-d-table-entry *compound-expression-table* constructor computed-hash)
     expression)
    expression))

(define (WALK-COMPOUND-EXPRESSIONS proc)
  (walk-table
   (lambda (key subtable)
     (ignore key)
     (walk-table
      (lambda (key entry-list)
	(ignore key)
	(walk
	 proc
	 entry-list))
      subtable))
   *compound-expression-table*))

(define (COUNT-IMPS-EXPRESSIONS)
  (let ((count 0))
    (walk-compound-expressions
     (lambda (e) (increment count)))
    count))

(define (WALK-COMPOUND-EXPRESSION-lists list-proc)
  (walk-table
   (lambda (key subtable)
     (ignore key)
     (walk-table
      (lambda (key entry-list)
	(ignore key)
	(list-proc entry-list))
      subtable))
   *compound-expression-table*))

;; For some reason this seems not to remove any of them anyway.  

(define (garbage-collect-expressions)
  (let ((wk (make-weak-set)))
    (walk-compound-expressions
     (lambda (e)
       (add-to-weak-set! wk e)))
    (walk-table
     (lambda (key subtable)
       (set (table-entry *compound-expression-table* key) '#f))
     *compound-expression-table*)
    (crawl (the-environment))
    (gc) 
    (set *compound-expression-table* (make-two-d-table '*compound-expression-table*))
    (walk-weak-set
     (lambda (e)
       (insert-in-compound-table
	(expression-constructor e)
	(expression-components e)
	e))
     wk)))
	 
