// request.c

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#pragma implementation "reqdelegate.h"
#endif

#include <String.h>
#include "localdefs.h"
#include "query.h"
#include "request.h"
#include "reqdelegate.h"

QueryLink::QueryLink(QueryItem *q) {
	element = q;
	element->ref();
	next = nil;
}

QueryLink::~QueryLink() { Resource::unref(element); }


QueryList::QueryList(const char *label) {
	listLabel = new String(label ? label : "");
	head = nil;
}

QueryList::~QueryList() {
	delete listLabel;
	start();
	while(current) {
		register QueryLink *next = current->next;
		delete current;
		current = next;
	}
}

QueryLink *
QueryList::tail() {
	start();
	while(current && current->next)
		current = current->next;
	return current;
}
	
void
QueryList::append(QueryItem *q) {
	register QueryLink *link = new QueryLink(q);
	if(!head)
		head = link;
	else
		tail()->next = link;
	current = link;
}

const char *
QueryList::label() { return *listLabel; }

////////

Request::Request(const QueryInfo *qinfo, RequestDelegate *delegate)
		: myDelegate(delegate) {
	init();
	if(qinfo) {
		if(qinfo->labelInfo)
			createLabelList(qinfo->labelInfo);
		if(qinfo->valueInfo)
			createValueList(qinfo->valueListLabel, qinfo->valueInfo);
		if(qinfo->choiceInfo)
			createChoiceList(qinfo->choiceInfo);
	}
	if(qinfo && qinfo->buttonInfo)
		createButtonList(qinfo->buttonInfo, qinfo->defaultResponse);
	else
		createButtonList(defaultInputButtonInfo, Yes);
}

Request::Request(const char *label, RequestDelegate *delegate)
		: myDelegate(delegate) {
	init();
	appendLabel(label);
	createButtonList(defaultInputButtonInfo, Yes);
}

void
Request::init() {
	labelList = nil;
	browserQuery = nil;
	valueList = nil;
	choiceList = nil;
	buttonList = nil;
	bell = false;
}

Request::~Request() {
	delete labelList;
	delete browserQuery;
	delete valueList;
	delete choiceList;
	delete buttonList;
	delete myDelegate;
}

void
Request::newLabelList() {
	if(!labelList) labelList = new QueryList;
}

void
Request::newValueList() {
	if(!valueList) valueList = new ValueList;
}

void
Request::newButtonList() {
	if(!buttonList) buttonList = new ButtonList;
}

void
Request::appendLabel(const char *lbl) {
	newLabelList();
	labelList->append(new QueryItem(lbl));
}

void
Request::appendLabel(const char *lblbeg, const char *lblend) {
	newLabelList();
	labelList->append(new QueryItem(lblbeg, lblend));
}

void
Request::addFileBrowser(const char* path, const char* suffixes) {
	delete browserQuery;						// only one at a time allowed
	browserQuery = new QueryFile(path, suffixes);
}

void
Request::appendButton(const char *lbl, Response response) {
	newButtonList();
	buttonList->append(new QueryButton(lbl, response));
}

void
Request::appendValue(const char *label, const char *value, CharCheckFun ccf) {
	newValueList();
	valueList->append(new QueryValue(label, value, ccf));
}

void
Request::appendValue(const char *label, int value, CharCheckFun ccf) {
	newValueList();
	valueList->append(new QueryValue(label, value, ccf));
}

void
Request::appendValue(const char *label, double value, CharCheckFun ccf) {
	newValueList();
	valueList->append(new QueryValue(label, value, ccf));
}

void
Request::appendValue(const char *label, int value, const Range& bounds) {
	newValueList();
	valueList->append(new RangedQueryValue(label, value, bounds));
}

void
Request::appendValue(const char *label, double value, const Range& bounds) {
	newValueList();
	valueList->append(new RangedQueryValue(label, value, bounds));
}

void
Request::appendChoice(const char *label, const char *list,
		unsigned states, boolean excl) {
	if(!choiceList) choiceList = new ChoiceList;
	choiceList->append(new QueryChoice(label, list, states, excl));
}

int
Request::retrieveValues(QueryReturn &value) {
	int found = 0;
	if(hasBrowser()) {
		value = *browserQuery->path();
		found = 1;
	}
	else if(hasValues()) {
		valueList->start();
		value = *(valueList->item());
		found = 1;
	}
	return found;
}

int
Request::retrieveValues(QueryReturn *array, int nelements) {
	int valsReturned = 0;
	ValueArray v(array, nelements);
	if(hasBrowser()) {		// browser's return value is always first
		v[valsReturned] = *browserQuery->path();
		valsReturned++;
	}
	if(hasValues()) {
		for(valueList->start(); valueList->more() && valsReturned < nelements; 
				valueList->next(), ++valsReturned) {
			v[valsReturned] = *(valueList->item());
		}
		valsReturned++;
	}
	return valsReturned;
}

int
Request::retrieveChoices(QueryReturn &choice) {
	int found = 0;
	if(hasChoices()) {
		choiceList->start();
		choice = *(choiceList->item());
		found = 1;
	}
	return found;
}

int
Request::retrieveChoices(QueryReturn *array, int nelements) {
	int valsReturned = 0;
	if(hasChoices()) {
		ValueArray v(array, nelements);
		for(choiceList->start();
				choiceList->more() && valsReturned < nelements; 
				choiceList->next(), ++valsReturned) {
			v[valsReturned] = *(choiceList->item());
		}
		valsReturned++;
	}
	return valsReturned;
}

void
Request::createLabelList(QueryLabelInfo *qlist) {
	for(QueryLabelInfo *i = qlist; i->label != nil; i++) {
		appendLabel(i->label);
	}
}

void
Request::createButtonList(QueryButtonInfo *qbilist, Response deflt) {
	buttonList = new ButtonList(nil, deflt);
	for(QueryButtonInfo *i = qbilist; i->label != nil; i++) {
		appendButton(i->label, i->response);
	}
}

// this method will be split into submethods once testing is complete

void
Request::createValueList(const char *vlistlabel, QueryValueInfo *qvilist) {
	valueList = new ValueList(vlistlabel);
	for(QueryValueInfo *i = qvilist; i->label != nil; i++) {
		if(i->check == nil) {	// means we are using range for values
			double low = 0, high = 0;
			if(!i->valueRange) {
				fprintf(stderr, "Error in bounded value item \"%s\": null range string.\n", i->label);
				exit(1);
			}
			sscanf(i->valueRange, "%lf|%lf", &low, &high);
			Range bounds(low, high);
			if(CharCheck::isIntegerString(i->defaultVal))
				appendValue(i->label, atoi(i->defaultVal), bounds);
			else if (CharCheck::isDoubleString(i->defaultVal))
				appendValue(i->label, atof(i->defaultVal), bounds);
			else {
				fprintf(stderr, "Error in bounded value item \"%s\": non-numeric default value \"%s\".\n", i->label, i->defaultVal);
				exit(1);
			}
		}
		else
			appendValue(i->label, i->defaultVal, i->check);
	}
}

void
Request::createChoiceList(QueryChoiceInfo *qcilist) {
	for(QueryChoiceInfo *i = qcilist; i->label != nil; i++) {
		appendChoice(i->label, i->choiceLabels, i->initialStates, i->exclusive);
	}
}

boolean
Request::print(FILE *out) {
	boolean status = true;
	if(hasLabels()) {
		for(labelList->start(); labelList->more(); labelList->next())
			fprintf(out, "%s\n", labelList->item()->label());
	}
	if(hasValues() || hasChoices()) {
		fprintf(out, "\nThis dialog cannot be viewed on a tty yet,\n");
		fprintf(out, "so this operation will return <CANCEL>.\n\n");
		status = false;
	}
	fflush(out);
	return status;
}

boolean
Request::checkValues() {
	return myDelegate ? myDelegate->checkValues(*this) : true;
}

//**********

MessageRequest::MessageRequest(const char *msg, const char *msg2, 
		const char *msg3) {
	if(msg) appendLabel(msg);
	if(msg2) appendLabel(msg2);
	if(msg3) appendLabel(msg3);
}

//**********

AlertRequest::AlertRequest(const char* m1, const char* m2, const char* m3,
	const char* btn) : MessageRequest(m1, m2, m3) { init(btn); }

void
AlertRequest::init(const char* btn) {
	appendButton(btn, Yes);
	setBell(true);
}

//******

ConfirmRequest::ConfirmRequest(const char *m1, const char *m2, const char *m3,
	  Response r, const char* b1, const char* b2) : MessageRequest(m1, m2, m3) {
	init(r, b1, b2);
}

void
ConfirmRequest::init(Response r, const char* b1, const char* b2) {
	buttonList = new ButtonList(nil, r);
	appendButton(b1, Yes);
	appendButton(b2, Cancel);
	setBell(true);
}

//******

ChoiceRequest::ChoiceRequest(const char *m1, const char *m2, const char *m3,
	Response r, const char* b1, const char* b2, const char* b3)
		: MessageRequest(m1, m2, m3) {
	init(r, b1, b2, b3);
}

void
ChoiceRequest::init(Response r, const char* b1, const char* b2, const char* b3)
{
	buttonList = new ButtonList(nil, r);
	appendButton(b1, Yes);
	appendButton(b2, No);
	appendButton(b3, Cancel);
	setBell(true);
}

//******

FileRequest::FileRequest(const char* title, const char* directory,
		const char* suffixes, RequestDelegate *delegate)
		: Request(title, delegate) {
	addFileBrowser(directory, suffixes);
}

//******

QueryButtonInfo defaultInputButtonInfo[] = {
	{ "confirm", Yes },
	{ "cancel", Cancel },
	{ nil }
};

