// sound.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "application.h"
#include "controller.h"
#include "converter.h"
#include "interpolater.h"
#include "sound.h"
#include "soundheader.h"
#include "request.h"
#include "smpte.h"
#include "sndconfig.h"
#include "statusaction.h"
#include "typeconvert.h"
#ifdef NO_TEMPLATES
#include <InterViews/action.h>		/* sgi native CC cant handle templates */
	declareActionCallback(Sound)
	implementActionCallback(Sound)
#else
#include "actioncallback.h"
#endif

// storage for public "class methods" -- macros in sndconfig.h

int Sound::default_SampleRate = SOUND_DEFAULT_RATE;

DataType Sound::default_DataType = SOUND_DEFAULT_TYPE;

char* Sound::default_FileSuffix = SOUND_SUFFIX;

// class methods

const char*
Sound::defaultFileSuffix() { return default_FileSuffix; }

// ctors, etc.

Sound::Sound(double duration, int srate, int nchans, DataType type)
	: Data(type, round(duration*srate), nchans), sr(srate) {
}

Sound::Sound(int len, int srate, int nchans, DataType type)
	: Data(type, len, nchans), sr(srate) {
}

// these protected constructors are used only with Sound::clone()

Sound::Sound(const Sound *s, const Range &selection)
	: Data(s, selection), sr(s->sr) {
}

Sound::Sound(const Sound *s, const Range &selection, const Range &chans)
	: Data(s, selection, chans), sr(s->sr) {
}

Sound::~Sound() {
}

Data *
Sound::newData() { return new Sound(this); }

Data *
Sound::newData(int length) { return new Sound(this, length); }

Data *
Sound::clone(const Range &selection) { return new Sound(this, selection); }

Data *
Sound::clone(const Range &selection, const Range &chans) {
	return new Sound(this, selection, chans);
}

void
Sound::copyFrom(const Data *src) {
	// if copying data with different max sample values
	if(valueLimit() != src->valueLimit() && dataType() < FloatData) {
		boolean wasSetTo = deferRescan(true);
		if(wasSetTo == true)
			((Sound *)src)->checkValues();	// scan if checking was turned off
		boolean sourceIsFloat = src->dataType() >= FloatData;
		double scalingFactor = valueLimit() /
			(sourceIsFloat ? src->maxValue() : src->valueLimit());
		Application::inform("Rescaling for copy...");
		deferRescan(wasSetTo);
		copyRescaledFrom(src, scalingFactor, sourceIsFloat);
	}
	else
		Super::copyFrom(src);
}

double
Sound::duration() const {
	return length()/double(sr);
}

const char *
Sound::fileSuffix() const {
	return dataType() >= FloatData ? SOUND_FLOAT_SUFFIX : defaultFileSuffix();
}

double
Sound::peakAmp() {
	return maxValue();
}

int
Sound::changeSRate(int rate, boolean interp) {
	int status = false;
	if(rate != sr && rate > 0) {
		status = true;
		int oldrate = sr;
		sr = rate;
		if(interp) {	// copy sound and then transpose it back into original
			boolean was = deferRescan(true);
			Data* source = copyOf();
			source->ref();
			Interpolater interp(source, this, double(oldrate)/sr, true);
			if((status = interp.ok()) == true)
				interp.apply();
			Resource::unref(source);
			deferRescan(was);
		}
		Notify();
	}
	return status;
}

// base class method is protected -- made public in Sound class

int
Sound::changeDataType(DataType newType) {
	return Super::changeDataType(newType);
}

// protected methods

Header *
Sound::createHeader(DataFile *file, boolean isNew) {
	SoundHeader *hdr = SoundHeader::create(
		file,
		dataType(),
		sRate(),
		nChans(),
		isNew ? 0.0 : maxValue()	// only extract peak from existing file
	);
	if(!isNew) writeToHeader(hdr);
	return hdr;
}

void
Sound::readFromHeader(Header *h) {
	Super::readFromHeader(h);
	SoundHeader *header = (SoundHeader *) h;
	sr = header->sampleRate();
	double peak = header->peakAmp();
	if(peak != 0.0)
		setMaxValue(peak);
	else
		checkValues();
}

void
Sound::checkValues() {
	double max = scanForMaxValue();
	setMaxValue((max == 0.0) ? 1.0 : max);	// recheck peak amp
}

Range
Sound::limits(int, boolean) const {
	double peak = maxValue();
	return Range(-peak, peak);	// same for all chans
}

// special protected method for use by the Converters

const char *
Sound::getDataPointer() const {
	return rep->getAddressOfContiguousData();
}

int
Sound::play(Converter *converter, StatusAction* wantStop) {
	Sound *playMe = nil;
	int status = true;
	if(converter->hasPlayableFormat(this))
		playMe = this;
	else {
		Application::inform("Converting to playable format...");
		playMe = new Sound(
			duration(), sRate(), nChans(), converter->bestPlayableType()
		);
		playMe->ref();
		playMe->copyFrom(this);
	}
	status = playMe->doPlay(converter, wantStop);
	if(playMe != this)
		Resource::unref(playMe);
	return status;
}

int
Sound::doPlay(Converter *converter, StatusAction* wantStop) {
	if(converter->configure(this, Converter::Play)) {
		return converter->start(wantStop, nil);
	}
	return false;
}

int
Sound::record(Converter *converter, StatusAction* wantStop) {
	return doRecord(converter, wantStop);
}

int
Sound::doRecord(Converter *converter, StatusAction* wantStop) {
	if(converter->configure(this, Converter::Record)) {
		return converter->start(
			wantStop,
#ifdef NO_TEMPLATES
			new Sound_ActionCallback(this, &Sound::Notify)
#else
			new ActionCallback<Sound>(this, &Sound::Notify)
#endif
		);
	}
	return false;
}

void
Sound::rescale() {
	boolean wasSetTo = deferRescan(false);	// we need to rescan here
	if(wasSetTo == true)
		checkValues();	// check first if checking was turned off
	// rather than rescanning again after rescaling, defer check and
	// simply set the peak to valueLimit()
	double oldpeak = maxValue();
	if(oldpeak != 0.0) {
		deferRescan(true);
		setMaxValue(valueLimit());
		scaleSelf(valueLimit()/oldpeak);
		deferRescan(wasSetTo);
	}
}

Range
Sound::frameRange(RangeUnit units) const {
    return (units == FrameTimeUnit) ? Range(0.0, duration())
	: (units == FrameSmpteUnit) ? Range(0.0, SMPTE::frames(duration()) - 1)
	: Super::frameRange(units);
}

const char * 
Sound::frameRangeLabel(RangeUnit units) const {
	return (units == FrameTimeUnit) ? "Time in Seconds" :
		(units == FrameSmpteUnit) ? "SMPTE Frames" :
		(units == FrameUnit) ? "Sample Number"
		: Super::frameRangeLabel(units);
}

void
Sound::information(Controller *controller) {
	char str[128];
	AlertRequest request("Soundfile Information:");
	request.setBell(false);
	request.appendLabel("------------");
	request.appendLabel("Filename: ", controller->fileName());
	request.appendLabel("Sample Rate: ", pstring(str, sRate()));
	request.appendLabel("Duration (seconds): ",
		pstring(str, duration()));
	request.appendLabel("File Size (Mb): ",
		pstring(str, sizeInBytes()/1000000.0));
	request.appendLabel("NSamps: ", pstring(str, length()));
	request.appendLabel("NChans: ", pstring(str, nChans()));
	DataType type = dataType();
	static char* labels[] = {
		"unknown",
		"8-bit linear", "mu law", "16-bit linear", "floating point",
		"unknown"
	};
	request.appendLabel("Sample Format: ", labels[type]);
	request.appendLabel("Peak Amplitude: ", pstring(str, maxValue()));
	controller->handleRequest(request);
}

#ifdef sgi

#include "aiff_header.h"
#include "datafile.h"

// these kludges are needed since AIFF sounds have special i/o routines

int
Sound::read(DataFile *f, Header* header) {
	BUG("Sound::read()");
	if(header == nil)
		header = createHeader(f, true);
	if(((SoundHeader *) header)->headerType() != SoundHeader::Aifc)
		return Super::read(f, header);	// not AIFC file
	header->ref();
	int status = false;
	if(status = header->read(f)) {		// supposed to be =, not ==
		if(header->dataSize() == 0)
			Application::alert("Warning:  file contains 0 frames.");
		Resource::unref(rep);
		int len = f->readSize() /
			(header->nChans() * type_to_sampsize(header->dataType()));
		rep = DataRep::create(header->dataType(), max(len, 8), header->nChans());
		rep->ref();
		AIFFSoundHeader* ah = (AIFFSoundHeader *) header;
		char* data =  (char *) getDataPointer();
		long samps = ah->readSound(&data, long(len));
		if(samps == long(len)) {
			readFromHeader(header);
			boolean was = deferRescan(true);
			Notify();					// update views but dont rescan first
			deferRescan(was);
			modified(false);
		}
		else status = false;
	}
	Resource::unref(header);
	return status;
}

int
Sound::write(DataFile* f) {
	BUG("Sound::write(f)");
    Header* header = createHeader(f);
	if(((SoundHeader *) header)->headerType() != SoundHeader::Aifc)
		return write(f, header);	// not AIFC file
	header->ref();
    AIFFSoundHeader* aheader = (AIFFSoundHeader *) header;
    int status = false;
	if((status = aheader->write(f)) == true) {
		char* data =  (char *) getDataPointer();
		status = (aheader->writeSound(data, long(length())) == length());
	}
    Resource::unref(aheader);
    modified(modified() && (status != true));
	return status;
}

#endif /* sgi */

