// selector.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include <InterViews/canvas.h>
#include <InterViews/painter.h>
#include <InterViews/rubband.h>
#include <InterViews/rubline.h>
#include <InterViews/rubrect.h>
#include <InterViews/interactor.h>
#include <InterViews/perspective.h>
#include "selector.h"
#include "range.h"
#include "graph.h"
#include <stream.h>

class RubberFillRect : public RubberRect {
public:
	RubberFillRect(Painter *p, Canvas *c,
			 Coord x0, Coord y0, Coord x1, Coord y1, Coord ox=0, Coord oy=0) 
			 : RubberRect(p, c, x0, y0, x1, y1, ox, oy) {}
	virtual void Draw();
};

void 
RubberFillRect::Draw () {
    Coord x0, y0, x1, y1;

    if (!drawn) {
	GetCurrent(x0, y0, x1, y1);
	if (x0 == x1 || y0 == y1) {
	    output->Line(canvas, x0+offx, y0+offy, x1+offx, y1+offy);
	} else {
            output->FillRect(canvas, x0+offx, y0+offy, x1+offx, y1+offy);
	}
	drawn = true;
    }
}

//********

Selector::Selector(ScaledArea *s)
		: area(s), shown(new Perspective(*(area->GetPerspective()))) {
	insert_set = region_set = false;
	insertloc = regionloc = 0;
	indicator = nil;
	height = width = xstart = xend = 0;
	confined = true;
	cacheValid = false;
	currentXmin = area->leftEdge();
	currentXmax = currentXmin + area->currentWidth();
	currentYmin = 0;
	currentYmax = area->currentHeight();
	updateSelf();	// useful here??
}

Selector::~Selector() {
	Resource::unref(indicator);
	Resource::unref(shown);
}

inline boolean
Selector::atBeginning() {
	return insertloc == 0 || regionloc == 0;
}

inline boolean
Selector::atEnd() {
	return insertloc == shown->width || regionloc == shown->width;
}

// public interface

// the next two are usually driven by mouse event locations

void
Selector::markInsertPoint(Coord loc) {
	Coord qloc = quantize(loc);
	if(region_set) {
		region_set = false;
		indicator->Erase();
	}
	if(!insert_set) {
		newIndicator(qloc, region_set);
		insert_set = true;
	}
	xend = 0;
	drawIndicator(qloc);
}

void
Selector::markEditRegion(Coord loc) {
	Coord qloc = quantize(loc);
	if(insert_set) {		// convert line to rectangle
		Coord oldstart, dummy;
		setCoordsFromEditPoints(oldstart, dummy);
		insert_set = false;
		indicator->Erase();
		region_set = true;
		newIndicator(xstart = oldstart, region_set);
	}
	else if(!region_set) {		// create new rectangle
		region_set = true;
		newIndicator(xstart = qloc, region_set);
	}
	drawIndicator(qloc);
}

// set insert hairline based on given coordinate

void
Selector::setInsert(Coord loc) {
	confine(false);
	markInsertPoint(loc);		// sets hairline without checking bounds
	confine(true);
}

// set hilighted box based on coordinate pair (Range)

void
Selector::setRegion(const Range &r) {
	setInsert(r.intMin());		// internally unconfined
	selectEdit();
	confine(false);
	markEditRegion(r.intMax());
	confine(true);
}

// set insert hairline based on frame insert location

void
Selector::setInsertLocation(int loc) {
	insertloc = regionloc = loc;
	Coord start, dummy;
	setCoordsFromEditPoints(start, dummy);
	setInsert(start);
	cacheIsValid(true);
}

// set hilighted box based on starting and ending frames of given edit region

void
Selector::setEditRegion(const Range &r) {
	insertloc = r.intMin();
	regionloc = r.intMax();
	Coord start, end;
	setCoordsFromEditPoints(start, end);
	confine(false);
	markInsertPoint(start);
	markEditRegion(end);
	confine(true);
	cacheIsValid(true);
}

// highlight and select currently visible portion of displayed data

void
Selector::selectVisible() {
	setRegion(Range(currentXmin, currentXmax));
}

// highlight and select all data frames

void
Selector::selectAll() {
	setEditRegion(Range(0, shown->width - 1));
}

void
Selector::unSelect() {
	if(isSet()) {
		insert_set = region_set = false;
		indicator->Erase();
		setEditPointsFromCoords(0, 0);
		cacheIsValid(false);
	}
}

void
Selector::shift(int direction) {
	double grain = max(area->currentHGrain(), 1.0);
	int offset = roundUp(direction * grain);
	if(direction > 0 && !atEnd() || direction < 0 && !atBeginning())
		confine(false);
	if(insert_set) {
		drawIndicator(quantize(xstart + offset));
	}
	else if(region_set) {
		int width = xend - xstart;
		Coord start = (xstart < xend) ?
			quantize(min(xstart+offset, currentXmax-width+1)) :
			quantize(max(xstart+offset, currentXmin-width-1));
		Coord end = start + width;
		indicator->Erase();
		moveRegionTo(start, end);
	}
	confine(true);
}

void
Selector::changeWidth(int direction) {
	if(region_set) {
		double grain = max(area->currentHGrain(), 1.0);
		int offset = round(direction * grain);
		int width = int(grain);	// current width of single grain
		if(!atEnd())
			confine(false);
		Coord end = (xstart < xend) ? 
			quantize(max(xend + offset, xstart + width)) :
			quantize(min(xend - offset, xstart - width)); 
		indicator->Erase();
		moveRegionTo(xstart, end);
		confine(true);
	}
}

// set insert point to beginning of (old) edit region

void
Selector::collapseRegion(int direction) {
	if(region_set) {
		setInsert((direction > 0) ? max(xstart, xend) : min(xstart, xend));
	}
	cacheIsValid(direction < 0);	// invalid if insert pt moved
}

boolean
Selector::selectEdit() {
	if(isSet()) {
		if(cacheInvalid())
			setEditPointsFromCoords(xstart, xend);
		return true;
	}
	else
		return false;
}

int
Selector::insertPoint() {
	return insert_set ?
		insertloc : region_set ? min(insertloc, regionloc) : 0;
}

Range
Selector::editRegion() {
	Range region;
	if(region_set)
		region.set(min(insertloc, regionloc), max(insertloc, regionloc));
	return region;
}

void
Selector::updateSelf(Coord l, Coord b, Coord r, Coord t) {
	Canvas *c = area->GetCanvas();
	if(c != nil) {
		currentXmin = area->leftEdge();
		currentXmax = currentXmin + area->currentWidth();
		currentYmin = 0;
		currentYmax = c->Height() - 1;
		int newHeight = currentYmax - currentYmin;
		int newWidth = currentXmax - currentXmin;
		if(newHeight != height || newWidth != width || graphAdjusted()) {
			updateGraphics();
			height = newHeight;
			width = newWidth;
		}
		else redraw(c, l, b, r, t);
	}
	*shown = *area->GetPerspective();	// copy this
}

// protected members

Coord
Selector::quantize(Coord x) {
	double grain = max(area->currentHGrain(), 1.0);
	if(confined) x = max(min(x, currentXmax), currentXmin);
	return int(round((x-currentXmin)/grain) * grain + currentXmin);
}

void
Selector::newIndicator(Coord loc, boolean region) {
	Resource::unref(indicator);
	indicator = nil;
	Painter *p = area->getOutput();
	Canvas *c = area->GetCanvas();
	if(!region)
		indicator = new SlidingLine(p, c, 0, currentYmax,
			0, currentYmin, 0, 0);
	else
		indicator= new RubberFillRect(p, c, loc, currentYmax, loc, currentYmin);
	indicator->ref();
}

void
Selector::drawIndicator(Coord loc) {
	indicator->Track(loc, currentYmin);
	if(insert_set)
		xstart = loc;
	else
		xend = loc;	
	cacheIsValid(false);
}

void
Selector::moveRegionTo(const Coord start, const Coord end) {
	newIndicator(xstart = start, region_set);
	indicator->Track(xend = end, currentYmin);
	cacheIsValid(false);
}

boolean
Selector::graphAdjusted() {
	return !(*shown == *area->GetPerspective());
}

// this is called when associated ScaledArea is redrawn

void
Selector::updateGraphics() {
	Coord start, end;
	if(insert_set) {
		setCoordsFromEditPoints(start, end);
		newIndicator(0, region_set);
		drawIndicator(start);
	}
	else if(region_set) {
		setCoordsFromEditPoints(start, end);
		moveRegionTo(start, end);	// redraw in updated location
	}
}

void
Selector::redraw(Canvas *c, Coord l, Coord b, Coord r, Coord t) {
	if(isSet()) {
		indicator->SetCanvas(c);
		indicator->GetPainter()->Clip(c, l, b, r, t);
		indicator->Redraw();
		indicator->GetPainter()->NoClip();
	}
}

void
Selector::setEditPointsFromCoords(const Coord start, const Coord end) {
	int screenstart = shown->curx;
	double grain = area->currentHGrain();
	int edstart = end > 0 ? min(start, end) : start;
	insertloc = max(0, round(((edstart - currentXmin)/grain) + screenstart));
	if(end == start)
		region_set = false;
	else if(end) {
		int edend = max(start, end);
		regionloc = max(0, round(((edend - currentXmin)/grain - 1) + screenstart));
		regionloc = min(regionloc, shown->width - 1);
	}
}

void
Selector::setCoordsFromEditPoints(Coord& start, Coord& end) {
	int screenstart = area->GetPerspective()->curx;
	double grain = area->currentHGrain();
	start = round((insertloc - screenstart) * grain) + currentXmin;
	end = round((regionloc + 1 - screenstart) * grain) + currentXmin;
}
