// scale.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// The Scale class is the visual representation of the scale markings shown
// to the left and above all Graphs.  It contains a Message object containing
// the label, and a ScaleMarks object which implements the actual tickmarks.
// The subclasses VScale and HScale exist to allow methods to be redefined for
// horizontal vs. vertical display of the graphics.  The same is true of the
// internal VScaleMarks and HScaleMarks subclasses.
//
// Scales are always associated with a ScaledArea instance.

#ifndef SCALE_H
#ifdef __GNUG__
#pragma interface
#endif
#define SCALE_H


#include <InterViews/scene.h>
#include "margin.h"
#include "scaledarea.h"
#include "range.h"

class Font;
class Range;
class ScaleMarks;
class Perspective;
class VBox;
class VMessage;

class Scale : public MonoScene {
public:
	enum NumberDisplay { AsInteger, AsFloat };
	enum Position { IsAbove, IsBelow, ToLeft, ToRight };
public:
	Scale();
	virtual ~Scale();
	redefined void Update();
	redefined void Reconfig();
	// new to class
	virtual int topMargin()=0;
	virtual int bottomMargin()=0;
	virtual int Height();
	virtual int Width();
	virtual void setRange(const Range &range);
	virtual Range getRange();
	void setLabel(const char *name);
	NumberDisplay numberDisplay();
	void setNumberDisplay(NumberDisplay nd);
	void showScale(boolean show);
	const char *getLabel() const;
	void setSpacers(int top, int bottom);
	void setViewPerspective(Perspective *np);
protected:
	virtual Interactor* getPlaceHolder() = 0;
	
	VBox *interior;
	ScaleMarks *scalemarks;
	Interactor* placeHolder;
	VMessage *label;
	boolean scaleShown;
	int pad;				// margin around scale labels
};

class VScale : public Scale {
	typedef Scale Super;
public:
	VScale(const char *name, const Range& r,
		NumberDisplay nd=AsFloat, Position pos=ToLeft);
	virtual ~VScale() {}
	redefined void Reconfig();
	redefined int topMargin();
	redefined int bottomMargin();
private:
	void Init(const char *, const Range &, NumberDisplay, Position);
	redefined Interactor* getPlaceHolder();
};

class HScale : public Scale {
	typedef Scale Super;
public:
	HScale(const char *name, const Range& r,
		NumberDisplay nd=AsFloat, Position pos=IsAbove);
	virtual ~HScale() {}
	redefined int topMargin();
	redefined int bottomMargin();
private:
	void Init(const char *, const Range&, NumberDisplay, Position);
	redefined Interactor* getPlaceHolder();
};

//********

class ScaleMarks : public BorderedArea {
public:
	ScaleMarks(const Range &r, Scale::NumberDisplay nd);
	virtual ~ScaleMarks();
	redefined void Update();
	// new to class
	void setRange(const Range& range);
	Range getRange() { return scaleRange; }
	virtual int baseLocation() = 0;
	virtual int originVisible() { return scaleRange.includesZero(); }
	void setSpacers(int top, int bottom);
	void setNumberDisplay(Scale::NumberDisplay nd) {
		displayIntegers = (nd == Scale::AsInteger);
	}
protected:
	redefined void Resize();
	redefined void doAdjust();
	virtual void drawTicks(Coord begin, Coord end)=0;
	virtual int scaleLength()=0;
	int textMajorDimension(char *string);
	int textMinorDimension(char *string);
	Scale::NumberDisplay numberDisplay() {
		return displayIntegers ? Scale::AsInteger : Scale::AsFloat;
	}
	virtual int getTextMajorDimension(const Font *f, char *string)=0;
	virtual int getTextMinorDimension(const Font *f, char *string)=0;
	virtual Range getCurrentRange()=0;
	virtual void newRange() {}
	void setViewPerspective(Perspective *np);
	virtual int viewChanged()=0;
protected:
	struct Tick {			// each hatchmark is a "Tick" object
		int loc;
		int length;
		char *string;
		Tick(int l, int len, char *str=0) : loc(l), length(len), string(0) {
			setString(str);
		}
		~Tick() { delete string; }
		char *setString(char *str);
	};
	class TickList;
	friend TickList;
	class TickList {
		Tick **list;
		int maxTicks;
		int currentsize;
	public:
		TickList(int size) : maxTicks(size), currentsize(0) {
			list = new Tick *[maxTicks];
		}
		~TickList();
		int currentSize() { return currentsize; }
		void addTick(int l, int len, char *str) {
			list[currentsize++] = new Tick(l, len, str);
		}
		Tick *get(int index) { return index < currentsize ? list[index] : 0; }
	};
	TickList *tickList;		// array of Tick objects
	int totalTicks;
	int labelPad;			// space between prime tick labels
	Range scaleRange;
	Perspective *shown;
	Perspective *view;		// pointer to controlling perspective
	int topSpacer;
	int bottomSpacer;
	boolean displayIntegers;	// status flags
	boolean rangeChanged;
	static const int defaultLabelPad;
	static const int defaultBorderWidth;
private:
	friend Scale;
	void Init();
};

class VScaleMarks : public ScaleMarks {
	typedef ScaleMarks Super;
public:
	VScaleMarks(const Range &r, Scale::NumberDisplay nd, Scale::Position)
		: ScaleMarks(r, nd) { Init(); }
	virtual ~VScaleMarks() {}
	redefined void Reconfig();
	// new functions
	redefined int baseLocation() {
		return BorderedArea::bottomBorder() + bottomSpacer;
	}
protected:
	redefined void Redraw(Coord, Coord, Coord, Coord);
	redefined void drawTicks(Coord begin, Coord end);
	redefined int viewChanged();
	redefined int scaleLength();
	redefined Range getCurrentRange();
	redefined void newRange();
	redefined int getTextMajorDimension(const Font *f, char *string);
	redefined int getTextMinorDimension(const Font *f, char *string);
private:
	void Init();
};

class HScaleMarks : public ScaleMarks {
	typedef ScaleMarks Super;
public:
	HScaleMarks(const Range &r, Scale::NumberDisplay nd, Scale::Position pos)
		: ScaleMarks(r, nd), isAbove(pos == Scale::IsAbove) { Init(); }
	virtual ~HScaleMarks() {}
	redefined void Reconfig();
	// new functions
	redefined int topBorder() { return leftEdge(); }
	redefined int bottomBorder() { return margin()->right; }
	redefined int baseLocation() { return leftEdge() + topSpacer; }
protected:
	redefined void Redraw(Coord, Coord, Coord, Coord);
	redefined void drawTicks(Coord begin, Coord end);
	redefined int scaleLength();
	redefined int viewChanged();
	redefined Range getCurrentRange();
	redefined void newRange();
	redefined int getTextMajorDimension(const Font *f, char *string);
	redefined int getTextMinorDimension(const Font *f, char *string);
private:
	void Init();
	boolean isAbove;
};

#endif /* SCALE_H */
