// reson.c

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "localdefs.h"
#include "reson.h"
#include "query.h"
#include "request.h"
#include <math.h>

ResonFilter::ResonFilter(Data* output, double fgain)
		: SimpleFunction(output), gain(fgain) {}

Reson3::Reson3(Data* output, double centerFreq, double bandWidth, GainMode scfac)
		: ResonFilter(output, 1.0), scale(scfac) {
	initialize();
	set(centerFreq, bandWidth);
}

Modifier *
Reson3::create(DataEditor *de) {
	return nil;
}

void
Reson3::set(double centerFreq, double bandWidth) {
	coef2 = exp(-M_PI * 2.0 * bandWidth / sampRate());
	double temp = 1.0 - coef2;
	double c = coef2 + 1;
	coef1 = 4.0 * coef2/c * cos(2.0 * M_PI * centerFreq / sampRate());
	coef0 = (scale == NoRescale) ? 1.0 :
		(scale == UnityRMS) ? sqrt(temp/c*(c*c - coef1*coef1)) :
		temp * sqrt(1.0 - coef1*coef1/(4.0*coef2));
	coef0 *= gain;		// gain factor added
}

void
Reson3::clear() {
	for(int i=0; i<2; i++) past[i] = 0.0;
}

const QueryInfo *
Reson3::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Apply Resonant Filter to Selected Region:" },
		{ nil }
	};	
	static QueryValueInfo valueInfo[] = {
		{ "Center Frequency (Hz.):", "1000.0", CharCheck::posNumsOnly },
		{ "BandWidth (Hz.):", "100", CharCheck::posNumsOnly },
		{ "Gain Factor:", "1.00", CharCheck::numsOnly },
		{ nil }
	};	
	static QueryChoiceInfo choiceInfo[] = {
		{ "Filter gain mode:",
		  "|Peak Set To Unity|Unity RMS Gain|No Rescaling|",
		  UnityRMS, Yes },
		{ nil }
	};
	static QueryInfo info[] = {
		{ labelInfo, "", valueInfo, choiceInfo },
		{ nil }
	};
	return info;
}

boolean
Reson3::setValues(Request& request) {
	boolean status = true;
	const int nvals = 3;
	QueryValue vr[nvals];
	request.retrieveValues(vr, nvals);
	double centerfreq = vr[0];
	if(centerfreq > sampRate()/2.0) {	// cutoff freq check
		Application::alert("Center frequency must be between 0 and the Nyquist.");
		return false;
	}
	double bandwidth = vr[1];
	if(bandwidth <= 0.0) {				// bandwidth check
		Application::alert("Bandwidth must be a nonzero number.");
		return false;
	}
	gain = vr[2];
	QueryChoice c;
	request.retrieveChoices(c);
	scale = GainMode(int(c));
	set(centerfreq, bandwidth);
	return status;
}

double
Reson3::operator () (double input) {
	double out;
	out = coef0 * input + coef1 * past[0] - coef2 * past[1];
	past[1] = past[0];
	past[0] = out;
	return out;
}

// ********

Reson4::Reson4(Data* output, double centerFreq, double bandWidth)
		: ResonFilter(output, 1.0) {
	initialize();
	set(centerFreq, bandWidth);
}

Reson4::~Reson4() {}

void
Reson4::set(double centerFreq, double bandWidth) {
	coef0 = exp(-M_PI * bandWidth / sampRate());
	coef1 = 1.0 - coef0;
	coef1 *= gain;		// gain factor added
	coef2 = 2.0 * coef0 * cos(2.0 * M_PI * centerFreq / sampRate());
	coef3 = -coef0 * coef0;
}

void
Reson4::clear() {
	for(int i=0; i<2; i++) inPast[i] = outPast[i] = 0.0;
}

double
Reson4::operator () (double input) {
	double out;
	out = coef1*(input-coef0*inPast[1])+coef2*outPast[0]+coef3*outPast[1];
	inPast[1] = inPast[0];
	inPast[0] = input;
	outPast[1] = outPast[0];
	outPast[0] = out;
	return out;
}
