// WinPlot3D.cpp : implementation of the WinPlot3D class.
// Applies Plot3D for Windows 95/NT
//

#include "stdafx.h"
#include "WinPlot3D.h"

/////////////////////////////////////////////////////////////////////////////
// Plotting Class

// --------------------------------------------------------------
// constructor
WinPlot3D::WinPlot3D()
{
	m_pWnd=0;
	m_pApp=0;
	m_pDC=0;
	m_monochrome_color=RGB(0,0,0);	// black
	m_draw_rate=0;
	m_draw_counts=0;
	m_is_color=0;
	m_fh_cap=0;
	m_pAbortFlag=0;
	m_background_color = RGB(255,255,255);	// white
}

// --------------------------------------------------------------
// destructor
WinPlot3D::~WinPlot3D()
{
	if (m_pDC && m_pWnd) m_pWnd->ReleaseDC(m_pDC);	// be sure to release any device context
	if (m_fh_cap) fclose(m_fh_cap);
	m_pDC = 0;
	m_pWnd = 0;
	m_pApp = 0;
}

// --------------------------------------------------------------
// Called when pen movement is requested
void WinPlot3D::WinMoveTo(int xv,int yv)
{
	if (m_fh_cap) fprintf(m_fh_cap,"M=%6d,%6d\n",xv,yv);
	m_pDC->MoveTo(xv,yv);	// issue a move to windows
}

// --------------------------------------------------------------
// Called when pen drawing is requested
void WinPlot3D::WinDrawTo(int xv,int yv)
{
	if (m_fh_cap) fprintf(m_fh_cap,"D=%6d,%6d\n",xv,yv);
	if (m_is_color)
	{
		CPen* pOldPen;
		CPen Pen(PS_SOLID,1,m_monochrome_color);	// construct a pen with the desired color
		pOldPen = m_pDC->SelectObject(&Pen);	// select the pen into the device context
		m_pDC->LineTo(xv,yv);		   	// draw the colored line
        m_pDC->SelectObject(pOldPen);	// restore the original pen
	}
	else
	{
		// just draw the line if in monochrome mode
		m_pDC->LineTo(xv,yv);
	}

	// check for slow draw
	if (!m_draw_rate) return;
	m_draw_counts++;	// one more line drawn
	if (m_draw_counts < m_draw_rate) return;
	Sleep(1);	// one millisecond delay
	m_draw_counts=0;	// reset counter
}

// --------------------------------------------------------------
// Called when the pen color is to be changed
void WinPlot3D::WinColorTo(long color)
{
	if (m_fh_cap) fprintf(m_fh_cap,"C=%-12d\n",color);
	m_monochrome_color = color;	// just save it
}

// --------------------------------------------------------------
// justification 1=left, 2=center, 3=right
void WinPlot3D::WinPlotText(AXIS_PLOT_INFO* pAxisInfo,double degrees,int xv,int yv,CString text)
{
	BOOL bRC;
	double radians;

	if (m_pDC)
	{
		// setup font info
		LOGFONT logFont;
		memset((char*)&logFont,0,sizeof(logFont));
		strcpy(logFont.lfFaceName,pAxisInfo->font_name);
		logFont.lfHeight = pAxisInfo->font_height;
		logFont.lfWidth  = pAxisInfo->font_width;

		// check if need to invert text (don't draw text upside down)
		radians = Deg2Rad(degrees);
		if (cos(radians)<0.)
		{
			// need to invert text
			CFont extFont;

			// determine width and height of text at zero degrees
			bRC = extFont.CreateFontIndirect(&logFont);
			m_pDC->SelectObject(extFont);
			CSize textExt = m_pDC->GetTextExtent(text);

			// calculate opposite corner
			xv = (int)((double)xv + (double)(textExt.cx)*cos(radians) - (double)(textExt.cy)*sin(radians));
			yv = (int)((double)yv - (double)(textExt.cx)*sin(radians) - (double)(textExt.cy)*cos(radians));
			degrees += 180.; // add 180 degrees
			if (degrees > 360.) degrees -= 360.;
		}

		// create a rotated font
		CFont rotFont;
		logFont.lfEscapement = (long)(degrees*10.);	// tenths of degree
		bRC = rotFont.CreateFontIndirect(&logFont);

		// output to redraw file
		if (m_fh_cap) fprintf(m_fh_cap,"T=%6d,%6d,%ld,%ld,%ld,%s %s\n",
						xv,yv,logFont.lfHeight,logFont.lfWidth,logFont.lfEscapement,logFont.lfFaceName,(LPCTSTR)text);

		// output the text
		m_pDC->SelectObject(rotFont);
		// ### show text in m_line_color
		m_pDC->TextOut(xv,yv,text);
	}
}

// --------------------------------------------------------------
// Stash away the pointer to our window and application
void WinPlot3D::SetWnd(CWnd* pWnd,CWinApp* pApp)
{
	m_pWnd = pWnd;
	m_pApp = pApp;
}

// --------------------------------------------------------------
// Stash away the pointer to the abort flag
// if null pointer, don't allow user abort
void WinPlot3D::SetAbortFlag(int* pAbortFlag)
{
	m_pAbortFlag = pAbortFlag;
}

// --------------------------------------------------------------
// Set the drawing rate: 0=off (normal speed)
void WinPlot3D::SetDrawRate(long draw_rate)
{
	m_draw_rate = draw_rate;
}

// --------------------------------------------------------------
// sets the name of the capture file
void WinPlot3D::SetFileName(CString fn)
{
	m_fn_cap = fn;
}

// --------------------------------------------------------------
// Call-back routine gets called when the plotting starts
// We need to get a device context and stash it for our use.
// We will also clear the screen at this time.
void WinPlot3D::WinBegPlot()
{
	if (!m_pWnd) return;

	// open the capture file
	m_fh_cap = fopen((LPCTSTR)m_fn_cap,"w+");
	if (m_fh_cap)
	{
		if (m_is_color)
			fprintf(m_fh_cap,"%s %s\n",PLOT3D_FILE_SIG,PLOT3D_COLOR_MODE);
		else
			fprintf(m_fh_cap,"%s %s\n",PLOT3D_FILE_SIG,PLOT3D_MONO_MODE);
	}

	// get window size
	RECT rect;
	m_pWnd->GetClientRect(&rect);
	CSize wsize = CSize(rect.right-rect.left, rect.bottom-rect.top);

	// get device context
	m_pDC = m_pWnd->GetDC();
	if (m_pDC==0) return;
	m_pDC->SetBkColor(m_background_color);
	m_pDC->SetMapMode(MM_ISOTROPIC);
	m_pDC->SetWindowOrg(0,wsize.cy);
	m_pDC->SetWindowExt(wsize);
	m_pDC->SetViewportExt(wsize);
	m_pDC->ScaleViewportExt(1,1,-1,1);	// invert y coordinate

	// clear the screen
	m_pDC->FillSolidRect(&rect,m_background_color);

	// select pen for monochrome mode
	if (!m_is_color)
	{
		CPen Pen(PS_SOLID,1,m_monochrome_color);	// construct a pen with the desired color
		m_pDC->SelectObject(&Pen);					// select the pen into the device context
	}
	fprintf(m_fh_cap,"S=%ld,%ld\n",m_background_color,m_monochrome_color);
}

// --------------------------------------------------------------
// Call-back routine gets called when the plotting is complete.
// Be sure to release the device context.
void WinPlot3D::WinEndPlot()
{
	// release device context
	if (m_pWnd && m_pDC) m_pWnd->ReleaseDC(m_pDC);
	m_pDC = 0;

	// close the capture file
	if (m_fh_cap) fclose(m_fh_cap);
	m_fh_cap = 0;
}

// --------------------------------------------------------------
// check for user abort request
// returns: 0=no, 1=user has requested plot abort
int WinPlot3D::IsAbortRequested()
{
	if (m_pApp==0) return(0);

	// keep window messages pumping to allow plot cancellation
/*
	MSG msg;
	if ( ::PeekMessage( &msg, NULL, 0, 0, PM_NOREMOVE ) ) 
	{ 
		if ( !m_pApp->PumpMessage( ) ) 
		{ 
			::PostQuitMessage(0); 
		} 
	}

	// check if user aborted plot
	if (m_pAbortFlag && *m_pAbortFlag)
		return(1);
*/
	return(0);
}

/*	<><><><><><><><><><><><><>  WinPlot3D.cpp  <><><><><><><><><><><><><><> */
