#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_add	WITH_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
)
/*
Sets z = x + y and returns z.  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y), zp = mp_ptr(z);
    mp_base_type	b;
    mp_expt_type	z_expt, exp_diff;
    mp_length		t, guard, total, dist, i;
    mp_bool		must_norm, same_sign;
    mp_acc_index	acc_index;
    mp_digit_ptr_type	p;
    mp_int		c;


    DEBUG_BEGIN(DEBUG_ADD);
    DEBUG_PRINTF_1("+add {\n");
    DEBUG_1("x = ", xp);
    DEBUG_1("y = ", yp);

    /*
    Check that x, y, & z have compatible parameters.
    */

    mp_check_3("mp_add", xp, yp, zp);


    /*
    Extract base and number of digits from z.
    */

    b = mp_b(zp);
    t = mp_t(zp);


    /*
    Check for x or y zero.
    */

    if (mp_is_zero(xp))
    {
	mp_copy_ptr(yp, zp);

	DEBUG_1("-} z = ", zp);

	DEBUG_END();
	return z;
    }

    if (mp_is_zero(yp))
    {
	mp_copy_ptr(xp, zp);
	DEBUG_1("-} z = ", zp);
	DEBUG_END();
	return z;
    }
    


    same_sign = mp_sign(xp) == mp_sign(yp);
    exp_diff = mp_expt(xp) - mp_expt(yp);
    dist = int_abs(exp_diff);


    must_norm = TRUE;

    if (exp_diff == 0)
    {
	/*
	Same exponent, so compare signs, then digits if necessary.
	*/

	if (!same_sign)
	{
	    register mp_int	dig_diff;

	    if ((dig_diff = mp_comp_digits(xp, yp)) == 0)
	    {
		mp_set_sign(zp, 0);
		DEBUG_PRINTF_1("-} z = 0\n");
		DEBUG_END();
		return z;
	    }

	    if (dig_diff > 0)
	    {
		/*
		Make y > x by swapping x and y.
		*/

		mp_float	temp_float;
		mp_ptr_type	temp_ptr;

		mp_swap_vars(x, y, temp_float);
		mp_swap_vars(xp, yp, temp_ptr);
	    }
	}

    }

    else
    {

	if (exp_diff > 0)
	{
	    /*
	    Make y > x by swapping x and y.
	    */

	    mp_float		temp_float;
	    mp_ptr_type		temp_ptr;

	    mp_swap_vars(x, y, temp_float);
	    mp_swap_vars(xp, yp, temp_ptr);
	}

	/*
	Now expt(y) > expt(x).
	*/

	if (dist > t)
	{
	    /*
	    The digits of x are less significant than all the guard digits.
	    If we are truncating or rounding simply, we can just ignore x.
	    Otherwise, if we are rounding up and subtracting (i.e. same_sign
	    == FALSE), or we are rounding down and adding (i.e. same_sign ==
	    TRUE), then the result would not be changed by rounding, so in
	    this case also x is ignored.
	    */

	    if (round == MP_TRUNC || round == MP_RND ||
		    (same_sign? round == MP_RND_DOWN: round == MP_RND_UP))
	    {
		/*
		x is negligible compared with y.
		*/

		mp_copy_ptr(yp, zp);
		DEBUG_1("-} z = ", zp);
		DEBUG_END();
		return z;
	    }

	    /*
	    Here directed rounding must occur.  Either we are adding and
	    rounding up, or we are subtracting and rounding down.  A slight
	    kludge is used to ensure the directed rounding.  The distance
	    between x & y is set to t so that in the addition/subtraction
	    x is copied into the guard digits.  mp_nzr() will then perform
	    the rounding if any of the guard digits are non-zero.
	    */

	    dist = t;
	}


	/*
	When we are truncating, normalization will not be needed if we are
	adding and there will not be a carry past the first digit, or we
	are subtracting and the first digit cannot become zero because of
	a borrow from the second digit.
	*/

	else if (round == MP_TRUNC && (same_sign? mp_digit(yp, 0) < b - 1:
					   mp_digit(yp, 0) > 1))

	    must_norm = FALSE;

    }


    /*
    Compute number of guard digits needed (now dist <= t).
    Brent adds one extra guard digit to dist if same_sign is true, but I can't
    see any good reason why.  Also he uses 1 guard digit if truncating, but
    this is unnecessary in the special case when normalization is not needed.
    */

    guard = (round == MP_TRUNC)? must_norm: dist;
    total = t + guard;


    /*
    Set p to point to output digits - if normalization not needed (i.e. no
    guard digits are used), point p directly at z, otherwise allocate storage.
    If accumulator storage is allocated, the pointers to the floats x, y, and
    z may need to be reset.
    */

    DEBUG_PRINTF_2("must_norm = %d\n", must_norm);

    if (must_norm)
    {
	mp_change_up();

	mp_acc_digit_alloc(total, acc_index);
	p = mp_acc_digit_ptr(acc_index);

	if (mp_has_changed())
	{
	    xp = mp_ptr(x);
	    yp = mp_ptr(y);
	    zp = mp_ptr(z);
	}

	mp_change_down();
    }

    else
	p = mp_digit_ptr(zp, 0);

#define z_dig(i)	p[i]
#define z_ptr(i)	(p + (i))


    DEBUG_PRINTF_2("dist = %d\n", dist);

    z_expt = mp_expt(yp);

    /*
    Zero trailing digits.
    */

    for (i = total - 1; i >= t + dist; i--)
	z_dig(i) = 0;


    if (same_sign)
    {    
	/*
	Perform addition.
	*/

	for (; i >= t; i--)
	    z_dig(i) = mp_digit(xp, i - dist);

	for (c = 0; i >= dist; i--)
	    if ((c += mp_digit(yp, i) + mp_digit(xp, i - dist)) < b) 
	    {
		/*
		No carry generated.
		*/
		
		z_dig(i) = c;
		c = 0;
	    }

	    else
	    {
		/*
		Carry generated.
		*/
				
		z_dig(i) = c - b;
		c = 1;
	    }
    
	for (; i >= 0; i--)
	{
	    if ((c += mp_digit(yp, i)) < b)
	    {
		z_dig(i) = c;
    
		/*
		No carry possible.
		*/
		
		c = 0;
		mp_dig_copy(z_ptr(0), mp_digit_ptr(yp, 0), i);
    
		break;
	    }
    
	    /*
	    c - b is now zero.
	    */

	    z_dig(i) = 0;
	    c = 1;
	}
    
	if (c)
	{
	
	    /*
	    Must shift right as carry is off end.
	    */
	
	    mp_shift_r(z_ptr(0), 1, total - 1);
	    z_dig(0) = 1;
	
	    z_expt++;
	}
    
    }

    else
    {
	/*
	Perform subtraction.
	*/

	for (c = 0; i >= t; i--)
	{
	    if ((z_dig(i) = c - mp_digit(xp, i - dist)) < 0)
	    {
		/*
		Borrow generated.
		*/

		z_dig(i) += b;
		c = -1;
	    }

	    else
		c = 0;
	}

	for (; i >= dist; i--)
	{
	    c += mp_digit(yp, i) - mp_digit(xp, i - dist);

	    
	    if (c < 0)
	    {
		/*
		Borrow generated.
		*/

		z_dig(i) = c + b;
		c = -1;
	    }

	    else
	    {
		/*
		No borrow generated.
		*/

		z_dig(i) = c;
		c = 0;
	    }
	}

	for (; i >= 0; i--)
	{
	    c += mp_digit(yp, i);

	    if (c >= 0)
	    {
		z_dig(i) = c;
	
		/*
		No carry possible.
		*/
		    
		mp_dig_copy(z_ptr(0), mp_digit_ptr(yp, 0), i);
	
		break;
	    }
      
	    z_dig(i) = c + b;
	    c = -1;
	}
    }

    if (must_norm)
    {
	mp_nzr(mp_sign(yp), z_expt, zp, z_ptr(0), guard);
	mp_acc_delete(acc_index);
    }
    else
    {
	mp_set_sign(zp, mp_sign(yp));
	mp_expt(zp) = z_expt;
    }

    DEBUG_1("-} z = ", zp);
    DEBUG_END();

    return z;
}
