/* This function subsumes the sac function
SAC MIUPQR 
*/

#include "defs.h"
#include "integer.e"
#include "poly.h"
#include "error.e"
#include "modint.e"

void
modpoly_quot_rem WITH_6_ARGS(
        t_handle,       pring,
	integer_big,	pdig,
	t_poly,	apoly,
	t_poly,	bpoly,
	t_poly *,	pqpoly,
	t_poly *,	prpoly
)
/*
** MODPOLY_QUOT_REM : modular polynomial quotient and remainder.
** apoly, bpoly are polynomials over Zpdig, pdig a prime integer,
** with bpoly non-zero.
** qpoly, rpoly are the unique polynomials over Zpdig such that 
** either bpoly | apoly, qpoly = apoly / bpoly and rpoly = 0
** or apoly = bpoly * qpoly + rpoly with deg( rpoly ) minimal.
*/
{
	block_declarations;
	t_poly	blead;
	t_poly	alead;
	t_int	bdeg;
	t_poly	bdashpoly;
	t_poly	q1poly;
	t_poly	qcoefft;
	t_poly	s;
	t_int	degdiff;
	t_handle		aph;
	t_handle		bph;
	t_handle		q1ph;
	t_poly	temp;
	t_poly	temp2;

	if ( m_poly_const( apoly ) )
	{
#ifdef DEBUG
        if (m_poly_not_const (apoly))
	    {
            error_internal ("modpoly_quot_rem: polys not lifted\n");
	    }
#endif /* DEBUG */

		if ( bpoly == 0 )
		{
			error_internal( "division by zero in modpoly_quot_rem" );
		}

		*pqpoly = modint_div( pdig, apoly, bpoly );
		if (! *pqpoly) 
			error_runtime(ERR_DIVIDE_BY_NON_UNIT, *pqpoly);

		*prpoly = 0;
		return;
	}

#ifdef DEBUG
	if ( m_poly_const( bpoly ) )
	{
		/* apoly is a non-trivial polynomial; bpoly is constant */
        error_internal ("modpoly_quot_rem: polys not lifted\n");
	}
#endif /* DEBUG */

	/* apoly, bpoly both non-trivial (non-integral) polynomials */

	aph = m_poly_poly_to_handle( apoly );
	bph = m_poly_poly_to_handle( bpoly );

#ifdef DEBUG
    if ((m_poly_least_pvar(aph) != m_poly_least_pvar(bph))
     || (m_poly_princvar (aph)  != m_poly_princvar (bph)))
	{
        error_internal ("modpoly_quot_rem: polys not lifted\n");
	}
#endif /* DEBUG */

	bdeg = poly_deg( bpoly );
	blead = poly_z_lead_coefft( pring, bpoly );
	bdashpoly = poly_z_reductum( pring, bpoly );

	*prpoly = m_modpoly_incref( pring, apoly );
	*pqpoly = poly_z_zero_poly (pring, apoly);

	/*	compute quotient terms	*/

	while ( ! poly_z_is_zero_poly (pring, *prpoly ))
	{
		degdiff = poly_deg( *prpoly ) - bdeg;

		if ( degdiff < 0 )
		{
			/* deg( rpoly ) < deg( bpoly ) */
			/* end condition satisfied	 */

			goto EXIT;
		}

		/* continue, deg( rpoly ) >= deg( bpoly ) */

		alead = poly_z_lead_coefft( pring, *prpoly );
		modpoly_quot_rem( pring, pdig, alead, blead, &qcoefft, &s );
		m_modpoly_delref( pring, alead );

		if ( !poly_z_is_zero_poly (pring, s) )
		{
			m_modpoly_delref( pring, qcoefft );
			m_modpoly_delref( pring, s );

			goto EXIT;
		}

		m_modpoly_delref( pring, s );
		/* continue, alead == blead * qcoefft, s == 0 */

		m_poly_create_empty( &q1ph, m_poly_princvar( aph ),
                                             m_poly_least_pvar(aph), 1 );
		q1poly = m_poly_handle_to_poly( q1ph );

		m_poly_coefft( q1ph, 0 ) = qcoefft;
		m_poly_expt( q1ph, 0 ) = degdiff;

		temp = *pqpoly;
		*pqpoly = modpoly_add( pring, pdig, temp, q1poly );
		m_modpoly_delref( pring, temp );

		temp2 = poly_z_reductum( pring, *prpoly );
		m_modpoly_delref( pring, *prpoly );

		temp = modpoly_mult( pring, pdig, bdashpoly, q1poly );
		m_modpoly_delref( pring, q1poly );

		*prpoly = modpoly_subtract( pring, pdig, temp2, temp );
		m_modpoly_delref( pring, temp );
		m_modpoly_delref( pring, temp2 );
	}

EXIT:	;

	m_modpoly_delref( pring, blead );
	m_modpoly_delref( pring, bdashpoly );

	return;
}

