/******************************************************************************
  anf_elt_simplify.c                                                           
 
  This file contains:
  anf_elt_simplify
  anf_elt_simplify_sub_a
  anf_elt_simplify_sub_b

******************************************************************************/

#include "kant.h"
#include "z.e"
#include "integer.e"
#include "anf.h"

static void anf_elt_simplify_sub_a P_((order, anf_elt, integer_big *, integer_big *));
static void anf_elt_simplify_sub_b P_((order, anf_elt, integer_big, integer_big));

anf_elt
anf_elt_simplify WITH_2_ARGS(
	order,		ord,
	anf_elt,	alpha
)
/*******************************************************************************
 
Description:
 
	Simplifies an algebraic number. This means that the result is an 
        algebraic number whose coefficients have no denominators (if they are
        anf_elts, too). This function also might return an integer_big.
 
        This routine should be called if it seems to be necessary, for example
        output routines should use it. It might make sense to call it after
        using anf_div.
         

 
Calling sequence:
 
	gamma = anf_elt_simplify(ord,alpha);
 
      	order       ord      = t_handle of order 
	anf_elt	    alpha    = algebraic number to be simplified
      	anf_elt     gamma    = algebraic number simplified
      
 
History:
 
	92-05-14 JS   	static definitions taken out again (Convex!)
	92-04-13 JS   	static definitions
	92-03-26 JS   	check whether anf_elt_is_integer
	92-03-24 JS   	no anf_elt_move, componentwise simplify instead
	92-02-10 JS    	case anf_elt_is_con
	91-12-18 KW    	written
 
*******************************************************************************/
{
	block_declarations;

        void anf_elt_simplify_sub_a();
        void anf_elt_simplify_sub_b();
                          
        order		ordcoef;
        anf_elt         alpha_s;
        integer_big     num;
        integer_big     den;
        integer_big     temp;
	integer_small	deg, i;
 
    
        if (anf_elt_is_integer(alpha)) return integer_incref(alpha);            

	if (anf_elt_is_integer_sub(ord,alpha,&temp)) return(temp);
  
	if (anf_elt_is_con(alpha)) return anf_elt_incref(alpha);
 
        ordcoef = order_coef_order(ord);
       	deg 	= order_rel_degree(ord);
                
/*
   copying alpha to alpha_s, simplifying coefficients
*/
        anf_elt_alloc(alpha_s, deg);
        anf_elt_den(alpha_s) = integer_incref(anf_elt_den(alpha));
        for (i=1; i<=deg; ++i)
        {
                anf_elt_coef(alpha_s, i) = 
		anf_elt_simplify(ordcoef, anf_elt_coef(alpha, i));
        }

/*
   common numerator and main denominator 
*/
        anf_elt_simplify_sub_a(ord,alpha_s,&num,&den);
        
        if (integer_sign(den) == -1)
        {
                temp = den;
                den = integer_negate(den);
		integer_delref(temp);
        }

        temp = num;
        num = integer_gcd(den,num);
        integer_delref(temp);
 
/*
    now simplification
*/
        anf_elt_simplify_sub_b(ord,alpha_s,num,den); 
                               
        temp = den;
        den = integer_div(den,num);
        integer_delref(temp);

        integer_delref(anf_elt_den(alpha_s));
        anf_elt_den(alpha_s) = den;
/*
    done
*/
/*         printf("\nSimplified:\n"); anf_elt_dump(ord, alpha_s);
        printf("\nalpha is now:\n"); anf_elt_dump(ord, alpha);
         printf("\n");*/
       return(alpha_s);
}   
                  


/*** internal ***/

static void
anf_elt_simplify_sub_a WITH_4_ARGS(
	order,		ord,
	anf_elt,	alpha,
        integer_big*,   num,
        integer_big*,   den
)            
/*******************************************************************************
 
Description:
 
	Computes common numerator and main denominator for all coefficients
	of alpha recursively.

 
History:
 
	92-03-31 JS    trivial case
 	92-03-26 JS/MJ integer_incref when initializing *num
	91-12-18 KW    written
 
*******************************************************************************/
{
	block_declarations;
 
        order		ordcoef;
        integer_big     tempa,tempb;
        integer_big     t_num,t_den;
	integer_small	deg;
	integer_small	i;          
/*
    trivial case
*/ 
        if (anf_elt_is_integer(alpha))
        {       
                *num = integer_incref(alpha);
                *den = 1;
                return;
        }

        ordcoef = order_coef_order(ord);
       	deg = order_rel_degree(ord);

        if (ring_type(ordcoef) == RING_Z)
        {
                *num = integer_incref(anf_elt_coef(alpha, 1));

                for (i=2;i<=deg;i++)
                {
                        tempa = *num;
                        *num = integer_gcd(tempa,anf_elt_coef(alpha,i));
                        integer_delref(tempa);
                }

                *den = integer_incref(anf_elt_den(alpha));

                return;
        }
        
        anf_elt_simplify_sub_a(ordcoef,anf_elt_coef(alpha,1),num,den);

        for (i=2;i<=deg;i++)
        {
                anf_elt_simplify_sub_a(ordcoef,anf_elt_coef(alpha,i),&t_num,&t_den);

                tempa = *num;
                tempb = *den;

                *num = integer_gcd(*num,t_num);
                *den = integer_lcm(*den,t_den);

                integer_delref(tempa);
                integer_delref(tempb);

                integer_delref(t_num);
                integer_delref(t_den);
        }

        tempa = *den;

        *den = integer_mult(tempa,anf_elt_den(alpha));

        integer_delref(tempa);
}


static void
anf_elt_simplify_sub_b WITH_4_ARGS(
	order,		ord,
	anf_elt,	alpha,
        integer_big,    num,
        integer_big,    den
)
/*******************************************************************************
 
Description:

	Has the task to appropriately move num and den into alpha.
                                                                  
 
History:
 
	92-03-31 JS    case anf_elt_coef(alpha, i) is an integer
	91-12-18 KW    written
 
*******************************************************************************/
{
	block_declarations;
 
        order		ordcoef;
        integer_big     tempa,tempb,tempc;
	integer_small	deg;
	integer_small	i;
	anf_elt         acoef;

        ordcoef = order_coef_order(ord);
       	deg = order_rel_degree(ord);

        tempa = integer_div(den,anf_elt_den(alpha));

        integer_delref(anf_elt_den(alpha));
        anf_elt_den(alpha) = 1;

        if (ring_type(ordcoef) == RING_Z)
        {
                for (i=1;i<=deg;i++)
                {
			tempb = anf_elt_coef(alpha,i);
			tempc = integer_mult(tempa,tempb);
                        anf_elt_coef(alpha,i) = integer_div(tempc,num);

			integer_delref(tempb);
			integer_delref(tempc);
                }
        }
        else
        {
                for (i=1;i<=deg;i++)
                {     
                        acoef = anf_elt_coef(alpha, i);
                        if (anf_elt_is_integer(acoef))
                        {
                                tempc = integer_mult(tempa, acoef);
	                        anf_elt_coef(alpha, i) = integer_div(tempc, num);

				integer_delref(acoef);
				integer_delref(tempc);
                        }
                        else
                        {
	                        anf_elt_simplify_sub_b(ordcoef, acoef, num, tempa);
                        }
                }
        }
        integer_delref(tempa);
}
